/*
 *  InputController.cpp
 *  SDLGame
 *
 *  Created by Mark Szymczyk on 10/25/06.
 *  Copyright 2006 Me and Mark Publishing. All rights reserved.
 *
 */

#include "InputController.h"

// Constructor
InputController::InputController(void)
{
    xAxisValue = AXIS_CENTERED;
    yAxisValue = AXIS_CENTERED;
    rotationXValue = 0;
}

// Destructor
InputController::~InputController(void)
{
    
}


// Accessors
SDL_Joystick* InputController::GetJoystick(void)
{
    return joystick;
}

void InputController::SetJoystick(SDL_Joystick* theStick)
{
    joystick = theStick;
}

Uint8* InputController::GetKeymap(void)
{
    return keymap;
}

void InputController::SetKeymap(Uint8* theMap)
{
    keymap = theMap;
}

Sint8 InputController::GetXAxisValue(void)
{
    return xAxisValue;
}

void InputController::SetXAxisValue(Sint8 axisValue)
{
    xAxisValue = axisValue;
}

Sint8 InputController::GetYAxisValue(void)
{
    return yAxisValue;
}

void InputController::SetYAxisValue(Sint8 axisValue)
{
    yAxisValue = axisValue;
}

Sint16 InputController::GetRotationXValue(void)
{
    return rotationXValue;
}

void InputController::SetRotationXValue(Sint16 rotationValue)
{
    rotationXValue = rotationValue;
}


// Event handling functions
void InputController::HandleKeyPress(SDL_Event* event)
{
    switch(event->key.keysym.sym) {
        case SDLK_LEFT:
            xAxisValue = MIN_X_AXIS_VALUE;
            break;
            
        case SDLK_RIGHT:
            xAxisValue = MAX_X_AXIS_VALUE;
            break;
            
        case SDLK_UP:
            yAxisValue = MAX_Y_AXIS_VALUE;
            break;
            
        case SDLK_DOWN:
            yAxisValue = MIN_Y_AXIS_VALUE;
            break;
            
        case SDLK_SPACE:
            //LaunchLaser(player1);
            break;
            
        default:
            break;
    }
}

void InputController::HandleKeyUp(SDL_Event* event)
{
    switch(event->key.keysym.sym) {
        case SDLK_LEFT:
            xAxisValue = AXIS_CENTERED;
            break;
            
        case SDLK_RIGHT:
            xAxisValue = AXIS_CENTERED;
            break;
            
        case SDLK_UP:
            yAxisValue = AXIS_CENTERED;
            break;
            
        case SDLK_DOWN:
            yAxisValue = AXIS_CENTERED;
            break;
            
        default:
            break;
    }
    
}

void InputController::HandleMouseMovement(SDL_Event* event)
{
    SetRotationXValue(event->motion.xrel);
}

void InputController::HandleMouseClick(SDL_Event* event)
{
    // The game has only one need for a button, which is to fire a laser,
    // so I have no need for this function. I can just fire the laser when
    // a mouse click occurs.
    
    // For your game it may be important to know what mouse button was clicked
    // and where the mouse was. The event parameter provides this information. The
    // button field tells you what button was clicked. SDL_BUTTON_LEFT for the left
    // button, SDL_BUTTON_RIGHT for the right button, and SDL_BUTTON_CENTER for the 
    // center button on a three-button mouse. The x and y fields tell you where the 
    // mouse was.
    
    Uint8 clickedButton;
    Uint16 clickX;
    Uint16 clickY;
    
    clickedButton = event->button.button;
    clickX = event->button.x;
    clickY = event->button.y;
    
    // Do what you need with the mouse click information.
    
}

void InputController::HandleMouseButtonUp(SDL_Event* event)
{
    // If your game needs to know when a mouse button was released,
    // you would call this function. The game for this book
    // has no need to handle button up events.

    Uint8 releasedButton;
    Uint16 releaseX;
    Uint16 releaseY;
    
    releasedButton = event->button.button;
    releaseX = event->button.x;
    releaseY = event->button.y;
    
    // Do what you need with the mouse release information.
}

void InputController::HandleJoystickAxisMovement(SDL_Event* event)
{    
    Uint8 movedAxis;
    movedAxis = event->jaxis.axis;
    
    Sint16 axisValue;
    Sint8 scaledAxisValue;
    axisValue = event->jaxis.value;
    scaledAxisValue = ScaleAxisValue(axisValue, MIN_RAW_AXIS_VALUE, MAX_RAW_AXIS_VALUE,
                                     MIN_X_AXIS_VALUE, MAX_X_AXIS_VALUE);
    
    if (movedAxis == X_AXIS)
        SetXAxisValue(scaledAxisValue);
    
    if (movedAxis == Y_AXIS)
        // Positive raw values on y axes indicate downward movement.
        // Reverse the y axis values so positive value indicates upward movement.
        SetYAxisValue(scaledAxisValue * -1);
    
}

void InputController::HandleJoystickButtonPress(SDL_Event* event)
{
    // The game has only one need for a button, which is to fire a laser,
    // so I have no need for this function. I can just fire the laser when
    // a joystick button press occurs.
    
    // Your game will probably use multiple buttons so you would want to know
    // what button was pressed. The event variable's button data member tells you
    // the button that was pressed. The event variable's button which variable
    // tells you what joystick was used to press the button.
    
    Uint8 pressedButton;
    pressedButton = event->jbutton.button;
    
    // Do what you need with the pressed button.
}

void InputController::HandleJoystickButtonUp(SDL_Event* event)
{
    // If your game needs to know when a joystick button was released,
    // you would call this function. The game for this book
    // has no need to handle button up events.
    
    Uint8 releasedButton;
    releasedButton = event->jbutton.button;
    
    // Do what you need with the released button.
    
}


// Functions to read the keyboard manually. I recommend using events to read the keyboard.
// Events are SDL's preferred way of reading the keyboard. But if you want to read the 
// keyboard manually, I have provided the code for you.
void InputController::ReadKeyboard(void)
{
    keymap = SDL_GetKeyState(NULL);
    
    // Set the x and y axis values to the center. When reading the keyboard manually,
    // you don't know when the player releases keys.
    SetXAxisValue(AXIS_CENTERED);
    SetYAxisValue(AXIS_CENTERED);
    
    // Now check the keys we're interested in.
    if (keymap[SDLK_LEFT]) {
        SetXAxisValue(MIN_X_AXIS_VALUE);
    }

    if (keymap[SDLK_RIGHT]) {
        SetXAxisValue(MAX_X_AXIS_VALUE);
    }

    if (keymap[SDLK_UP]) {
        SetYAxisValue(MAX_Y_AXIS_VALUE);
    }

    if (keymap[SDLK_DOWN]) {
        SetYAxisValue(MIN_Y_AXIS_VALUE);
    }
    
    
}


// Functions to read the mouse manually. I recommend using events to read mouse clicks
// and mouse movements.
void InputController::ReadMouse(void)
{
    // This function reads the absolute mouse position, which is useful
    // if you want to know where the player clicked.

    Uint8 buttonsPressed;
    int x;
    int y;
    
    // I care only about horizontal mouse movement, which I use to rotate the ship.
    buttonsPressed = SDL_GetMouseState(&x, &y);
    
    // SDL_GetMouseState() returns a bit mask that tells you what mouse
    // buttons were down when reading the mouse state. To test if a mouse button was
    // pressed, you would do a bitwise AND, such as the following
    //
    // if (buttonsPressed & SDL_BUTTON(X))
    //
    // where X is 1 for the left mouse button, 2 for the right button, and 3 for the
    // center button.
    
}

void InputController::ReadMouseRelative(void)
{
    // This function tells you how far the player moved the mouse.
 
    Uint8 buttonsPressed;
    int x;
    
    // I care only about horizontal mouse movement, which I use to rotate the ship.
    buttonsPressed = SDL_GetRelativeMouseState(&x, NULL);
    SetRotationXValue(x);
    
    // SDL_GetRelativeMouseState() returns a bit mask that tells you what mouse
    // buttons were down when reading the mouse state. To test if a mouse button was
    // pressed, you would do a bitwise AND, such as the following
    //
    // if (buttonsPressed & SDL_BUTTON(X))
    //
    // where X is 1 for the left mouse button, 2 for the right button, and 3 for the
    // center button.
    
}


// Functions to read the joystick manually. I recommend using events to read the joystick.
// Events are SDL's preferred way of reading the joystick. But if you want to read the 
// joystick manually, I have provided the code for you.
void InputController::ReadJoystick(void)
{
     SDL_JoystickUpdate();
     
     Sint16 rawX;
     Sint8 scaledX;
     
     rawX = SDL_JoystickGetAxis(GetJoystick(), X_AXIS);
     scaledX = ScaleAxisValue(rawX, MIN_RAW_AXIS_VALUE, MAX_RAW_AXIS_VALUE,
                                      MIN_X_AXIS_VALUE, MAX_X_AXIS_VALUE);
     SetXAxisValue(scaledX);

     Sint16 rawY;
     Sint8 scaledY;
     
     rawY = SDL_JoystickGetAxis(GetJoystick(), Y_AXIS);
     scaledY = ScaleAxisValue(rawY, MIN_RAW_AXIS_VALUE, MAX_RAW_AXIS_VALUE,
                              MIN_X_AXIS_VALUE, MAX_X_AXIS_VALUE);

     // Positive raw values on y axes indicate downward movement.
     // Reverse the y axis values so positive value indicates upward movement.
     SetYAxisValue(scaledY * -1);
     
     // Check for button presses
     /*
     int buttonTotal;
     int currentButton;
     bool buttonPressed;
     
     buttonTotal = SDL_JoystickNumButtons(GetJoystick());
     for (currentButton = 0; currentButton < buttonTotal; currentButton++) {
         buttonPressed = WasButtonPressed(currentButton);
         if (buttonPressed) {
             // Do what you need with the button
         }
     }
     */
}

bool InputController::WasButtonPressed(int button)
{
    Sint8 buttonPressed;
    
    buttonPressed = SDL_JoystickGetButton(GetJoystick(), button);
    if (buttonPressed == 1)
        return true;
    else
        return false;
}


// Utility functions
Sint8 InputController::ScaleAxisValue(Sint16 rawAxisValue, Sint16 minRawValue, Sint16 maxRawValue,
                     Sint8 minScaledValue, Sint8 maxScaledValue)
{
    // SDL returns raw axis values ranging from -32768 to 32767. I want to scale the values
    // so they range from -5 to 5, with -5 meaning the player moved the joystick all the way to
    // the left, 5 meaning the player moved the joystick all the way to the right and 0 
    // being the center with no movement. This function scales the raw axis values.
    
    // Because SDL's raw axis values range from -32768 to 32767, it is best to use a negative number
    // for minScaledValue and a positive number for maxScaledValue, with minScaledValue and
    // maxScaledValue having the same absolute value. If you did something like use 0 for minScaledValue
    // and 10 for maxScaledValue, you would have to modify this function.
    
    // Don't use too high or too low an absolute value
    // for minScaledValue and maxScaledValue. With a small value like 1, no movement will be registered
    // until the player moves the joystick a third of the way across either the x or y axis. With a large
    // value like 100, the joystick dead zone will be too small and movement will be detected when
    // the joystick is centered. Absolute values ranging from 3-10 would be best.
    
    Sint16 divisor;
    Sint8 scaledAxisValue;
    
    // The +1 provides a zone for extreme values. Without the +1 the player would have
    // to reach the minimum raw axis value to get the minimum scaled axis value.
    divisor = (maxRawValue - minRawValue) / (maxScaledValue - minScaledValue + 1);
    
    // Check for division by 0
    if (divisor == 0)
        return AXIS_CENTERED;
    
    scaledAxisValue = rawAxisValue / divisor;
    return scaledAxisValue;
}

