// PhysicsController.cp		by Mark Szymczyk

#ifndef PHYSICS_CONTROLLER
	#include "PhysicsController.h"
#endif

// Constructor
PhysicsController::PhysicsController(void)
{
	modelToControl = nil;
}

PhysicsController::~PhysicsController(void)
{
	
}

// Accessors
GameSpritePtr PhysicsController::GetModelToControl(void) 			
{ 
    return modelToControl; 
}

void PhysicsController::SetModelToControl(GameSpritePtr theModel) 	
{ 
    modelToControl = theModel; 
}


void PhysicsController::UpdatePhysics(InputControllerAction theAction, Vector2DPtr force, 
        double timeStep, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	if (modelToControl == nil)
        return;

	if (force == nil)
		return;
		
	if (currentLevel == nil)
		return;
		
	if (theTileList == nil)
		return;    
	
	Vector2D velocityVector = modelToControl->GetVelocity();
	double modelXVelocity = velocityVector.GetXComponent();
	double modelYVelocity = velocityVector.GetYComponent();
	
	double horizontalDistance;
	double verticalDistance;
	double greaterDistance;
			
	// For the direction the character moved, determine whether we
	// can move the amount of pixels we want to move in that particular 
	// direction.  If so, move.  If not, determine how far we can move
	// and move that number of pixels.
	switch(theAction) {
		case kMoveUp:
			// When moving up, the character's velocity will be negative.
			// The CanMoveUp() function takes a positive value so we must
			// multiply the velocity by -1.
			// The +1.0 rounds up the velocity so the creature doesn't 
			// penetrate a wall by a fraction of a pixel.
			verticalDistance = (modelYVelocity * timeStep * -1.0) + 1.0;
			
			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			if (verticalDistance > kTileHeight) {
				verticalDistance = (double)kTileHeight;
				velocityVector.SetYComponent(verticalDistance / timeStep * -1.0);// * timeStep * -1.0);
				velocityVector.SetXComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
			}
				
			if (CanMoveUp((short)verticalDistance, currentLevel, theTileList)){
				Move(timeStep);
			}
			else {
				// Move as far as we can

				// The HowFarUp() function returns a positive value so we must
				// multiply the result by -1.
				verticalDistance = (double) (HowFarUp() * -1.0);
				velocityVector.SetYComponent(verticalDistance);// * timeStep);
				velocityVector.SetXComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);
			}	
			break;

		case kMoveDown:
			// The +1.0 rounds up the velocity so the creature doesn't 
			// penetrate a wall by a fraction of a pixel.
			verticalDistance = (modelYVelocity * timeStep) + 1.0;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			if (verticalDistance > kTileHeight) {
				verticalDistance = (double)kTileHeight;
				velocityVector.SetYComponent(verticalDistance / timeStep); // * timeStep);
				velocityVector.SetXComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);			
			}
			
			if (CanMoveDown((short)verticalDistance, currentLevel, theTileList)) {
				Move(timeStep);
			}
			else { 
				// Move as far as we can
				verticalDistance = (double)HowFarDown();
				velocityVector.SetYComponent(verticalDistance);// * timeStep);
				velocityVector.SetXComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);
			}				
				
				break;
			
		case kMoveLeft:
			// When moving left, the character's velocity will be negative.
			// The CanMoveLeft() function takes a positive value so we must
			// multiply the velocity by -1.
			// The +1.0 rounds up the velocity so the creature doesn't 
			// penetrate a wall by a fraction of a pixel.
			horizontalDistance = (modelXVelocity * timeStep * -1.0) + 1.0;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			if (horizontalDistance > kTileWidth) {
				horizontalDistance = (double)kTileWidth;
				velocityVector.SetXComponent(horizontalDistance / timeStep * -1.0);// * timeStep * -1.0);
				velocityVector.SetYComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
			}				

			if (CanMoveLeft((short)horizontalDistance, currentLevel, theTileList)) {
				Move(timeStep);
			}
			else {
				// Move as far as we can
				
				// The HowFarLeft() function returns a positive value so we must
				// multiply the result by -1.
				horizontalDistance = (double) (HowFarLeft() * -1.0);
				velocityVector.SetXComponent(horizontalDistance);// * timeStep);
				velocityVector.SetYComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);			
			}
			break;
			
		case kMoveRight:
			// The +1.0 rounds up the velocity so the creature doesn't 
			// penetrate a wall by a fraction of a pixel.
			horizontalDistance = (modelXVelocity * timeStep) + 1.0;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			if (horizontalDistance > kTileWidth) {
				horizontalDistance = (double)kTileWidth;
				velocityVector.SetXComponent(horizontalDistance / timeStep);// * timeStep);
				velocityVector.SetYComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
			}
						
			if (CanMoveRight((short)horizontalDistance, currentLevel, theTileList)) {
				Move(timeStep);
			}
			else {
				// Move as far as we can
				horizontalDistance = (double) HowFarRight();
				velocityVector.SetXComponent(horizontalDistance);// * timeStep);
				velocityVector.SetYComponent(kStandingSpeed);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);						
			}
			
			break;

		case kMoveUpAndLeft:
			// Pick the higher of the horizontal and vertical distances 
            horizontalDistance = (modelXVelocity * timeStep * -1.0) + 1.0;
            verticalDistance = (modelYVelocity * timeStep * -1.0) + 1.0;
            
            if (horizontalDistance > verticalDistance)
            	greaterDistance = horizontalDistance;
            else
            	greaterDistance = verticalDistance;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			// The tile width and tile height are the same.  I arbitrarily
			// chose the width.
			if (greaterDistance > kTileWidth) {
				greaterDistance = (double)kTileWidth;
				velocityVector.SetXComponent(greaterDistance / timeStep);
				velocityVector.SetYComponent(greaterDistance / timeStep);
				modelToControl->SetVelocity(velocityVector);
			}
            	
            if (CanMoveUpAndLeft((short)greaterDistance, currentLevel, theTileList)) {
				Move(timeStep);
            }
			else {
				// Move as far as we can
				verticalDistance = (double) (HowFarUp() * -1.0);
				velocityVector.SetYComponent(verticalDistance);// * timeStep);
				horizontalDistance = (double) (HowFarLeft() * -1.0);
				velocityVector.SetXComponent(horizontalDistance);// * timeStep);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);			
			
            }
            
			break;

		case kMoveUpAndRight:
			// Pick the higher of the horizontal and vertical distances 
            horizontalDistance = (modelXVelocity * timeStep) + 1.0;
            verticalDistance = (modelYVelocity * timeStep * -1.0) + 1.0;

            if (horizontalDistance > verticalDistance)
            	greaterDistance = horizontalDistance;
            else
            	greaterDistance = verticalDistance;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			// The tile width and tile height are the same.  I arbitrarily
			// chose the width.
			if (greaterDistance > kTileWidth) {
				greaterDistance = (double)kTileWidth;
				velocityVector.SetXComponent(greaterDistance / timeStep);
				velocityVector.SetYComponent(greaterDistance / timeStep);
				modelToControl->SetVelocity(velocityVector);
			}
            
            if (CanMoveUpAndRight((short)greaterDistance, currentLevel, theTileList)) {
				Move(timeStep);
			}
            else {
				// Move as far as we can
				verticalDistance = (double) (HowFarUp() * -1.0);
				velocityVector.SetYComponent(verticalDistance);// * timeStep);
				horizontalDistance = (double) HowFarRight();
				velocityVector.SetXComponent(horizontalDistance);// * timeStep);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);			
            }
			break;
				
		case kMoveDownAndRight:
			// Pick the higher of the horizontal and vertical distances 
            horizontalDistance = (modelXVelocity * timeStep) + 1.0;
            verticalDistance = (modelYVelocity * timeStep) + 1.0;

            if (horizontalDistance > verticalDistance)
            	greaterDistance = horizontalDistance;
            else
            	greaterDistance = verticalDistance;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			// The tile width and tile height are the same.  I arbitrarily
			// chose the width.
			if (greaterDistance > kTileWidth) {
				greaterDistance = (double)kTileWidth;
				velocityVector.SetXComponent(greaterDistance / timeStep);
				velocityVector.SetYComponent(greaterDistance / timeStep);
				modelToControl->SetVelocity(velocityVector);
			}
            
            if (CanMoveDownAndRight((short)greaterDistance, currentLevel, theTileList)) {
				Move(timeStep);
            }
			else {
				// Move as far as we can
				verticalDistance = (double)HowFarDown();				
				velocityVector.SetYComponent(verticalDistance);// * timeStep);
				horizontalDistance = (double) HowFarRight();
				velocityVector.SetXComponent(horizontalDistance);// * timeStep);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);			
            }
            break;

		case kMoveDownAndLeft:
			// Pick the higher of the horizontal and vertical distances 
            verticalDistance = (modelYVelocity * timeStep) + 1.0;
            horizontalDistance = (modelXVelocity * timeStep * -1.0) + 1.0;

            if (horizontalDistance > verticalDistance)
            	greaterDistance = horizontalDistance;
            else
            	greaterDistance = verticalDistance;

			// Keep the sprite from moving more than one tile per frame.
			// This will keep the sprite from accidentally moving through walls.
			// The tile width and tile height are the same.  I arbitrarily
			// chose the width.
			if (greaterDistance > kTileWidth) {
				greaterDistance = (double)kTileWidth;
				velocityVector.SetXComponent(greaterDistance / timeStep);
				velocityVector.SetYComponent(greaterDistance / timeStep);
				modelToControl->SetVelocity(velocityVector);
			}
            
            if (CanMoveDownAndLeft((short)greaterDistance, currentLevel, theTileList)) {
				Move(timeStep);
			}
            else {
				// Move as far as we can
				verticalDistance = (double)HowFarDown();				
				velocityVector.SetYComponent(verticalDistance);// * timeStep);
				horizontalDistance = (double) (HowFarLeft() * -1.0);
				velocityVector.SetXComponent(horizontalDistance);// * timeStep);
				modelToControl->SetVelocity(velocityVector);
				Move(timeStep);			
			}
            break;
			
        case kAttack:
            //Stand();
            //Move(timeStep);
            break;
            
		case kNoMovement:
            Stand();			
			break;
				
		default:
            break;
	}

	// I had a serious error in the old version of the code.
	// I did the calculations like this:
	//
	// newPosition = oldPosition + (newVelocity * timeStep)
	// newVelocity = oldVelocity + (newAcceleration * timeStep)
	//
	// when it should be:
	// newPosition = oldPosition + (oldVelocity * timeStep)
	// newVelocity = oldVelocity + (oldAcceleration * timeStep)
	//
	// That's why CalculateVelocity() comes after Move() and
	// CalculateAcceleration() follows CalculateVelocity().
	
	CalculateVelocity(timeStep);
    CalculateAcceleration(force);

}


// Linear motion functions
Vector2D PhysicsController::CalculateCenterOfMass(void)
{
	// We just calculate the center of the sprite in this case.
	// If you want to use a more elaborate formula to calculate the
	// center of mass, substitute it in this function.
	
	Vector2D result;
	
	Vector2D positionVector = modelToControl->GetPosition();
	
	double xComponent;
	double yComponent;
	
	xComponent = positionVector.GetXComponent() + modelToControl->GetSpriteWidth();
	yComponent = positionVector.GetYComponent() + modelToControl->GetSpriteHeight();
	
	result.SetXComponent(xComponent);
	result.SetYComponent(yComponent);
	return result;
}

void PhysicsController::CalculateAcceleration(Vector2DPtr force)
{
	// acceleration = force / mass
	
	Vector2D newAcceleration;
	double xComponent = force->GetXComponent() / modelToControl->GetMass();
	double yComponent = force->GetYComponent() / modelToControl->GetMass();
	
	newAcceleration.SetXComponent(xComponent);
	newAcceleration.SetYComponent(yComponent);
	modelToControl->SetAcceleration(newAcceleration);
}

void PhysicsController::CalculateVelocity(double timeStep)
{
	// newVelocity = oldVelocity + (oldAcceleration * timeStep)

	// Call this function before calling CalculateAcceleration().
	
	Vector2D oldVelocity = modelToControl->GetVelocity();
    Vector2D accelerationVector = modelToControl->GetAcceleration();
	
	double xComponent = oldVelocity.GetXComponent() + (accelerationVector.GetXComponent() * timeStep);
	double yComponent = oldVelocity.GetYComponent() + (accelerationVector.GetYComponent() * timeStep);
	
	// This commented out code limits the character's speed
	// to keep the movement realistic.  When I tried it out, it
	// slowed the movement to a crawl so I commented it out.
	// If you need to limit the character's speed, remove the comments.
	
	if (xComponent > modelToControl->GetMaxSpeed())
		xComponent = modelToControl->GetMaxSpeed();

	if (xComponent < (modelToControl-> GetMaxSpeed() * -1.0))
		xComponent = modelToControl-> GetMaxSpeed() * -1.0;
		
	if (yComponent > modelToControl->GetMaxSpeed())
		yComponent = modelToControl->GetMaxSpeed();

	if (yComponent < (modelToControl-> GetMaxSpeed() * -1.0))
		yComponent = modelToControl-> GetMaxSpeed() * -1.0;
	
	Vector2D newVelocity;
    	
	newVelocity.SetXComponent(xComponent);
	newVelocity.SetYComponent(yComponent);
	modelToControl->SetVelocity(newVelocity);
}

void PhysicsController::Move(double timeStep)
{
	// newPosition = oldPosition + (oldVelocity * timeStep)

    // Call this function before calling CalculateVelocity().	
	
    Vector2D oldPosition = modelToControl->GetPosition();
    Vector2D newPosition;
	Vector2D velocityVector = modelToControl->GetVelocity();
	
	double xComponent = oldPosition.GetXComponent() + (velocityVector.GetXComponent() * timeStep);
	double yComponent = oldPosition.GetYComponent() + (velocityVector.GetYComponent() * timeStep);
	
	newPosition.SetXComponent(xComponent);
	newPosition.SetYComponent(yComponent);
	modelToControl->SetPosition(newPosition);
	
	// Update worldX and worldY for drawing the sprite
    modelToControl->SetWorldX((short)xComponent);
	modelToControl->SetWorldY((short)yComponent);	
}


// Angular motion functions
double PhysicsController::CalculateMomentOfInertia(GameSpritePtr model)
{
	// For a rectangular body, the moment of inertia equals
	// mass * (height squared + width squared) / 12
	double height = (double)model->GetSpriteHeight();
	double width = (double)model->GetSpriteWidth();
	
	double result;
	result = model->GetMass() * ((height * height) + (width * width)) / 12.0;
	return result;
}

void PhysicsController::CalculateAngularAcceleration(double torque)
{
	// angular acceleration = torque / moment of inertia
	
	double momentOfInertia;
	momentOfInertia = CalculateMomentOfInertia(modelToControl);
	
	double result;
	
	result = torque / momentOfInertia;
	modelToControl->SetAngularAcceleration(result);
}

void PhysicsController::CalculateLinearAcceleration(Vector2DPtr originVector)
{
	// Calculate linear acceleration from angular acceleration
	// It calculates the linear acceleration for a single point on the rotating body,
	// not the total linear acceleration
	
	// linear acceleration = (angular acceleration) * (perp vector from origin to the point)
	Vector2D perpVector;
	perpVector = Vector2D::GetPerpendicularVector(originVector);
    //perpVector = originVector->GetPerpendicularVector(originVector);
	
	double angularAccelerationValue;
	angularAccelerationValue = modelToControl->GetAngularAcceleration();
	
	Vector2D result;
	result = Vector2D::ScalarProduct(&perpVector, angularAccelerationValue);
    //result = result.ScalarProduct(&perpVector, angularAccelerationValue);
	modelToControl->SetAcceleration(result);
	
}

void PhysicsController::CalculateAngularVelocity(double timeStep)
{
	// new angular velocity = old angular velocity + (angular acceleration * time step)
	
	double result;
	
	result = modelToControl->GetAngularVelocity() + (modelToControl->GetAngularAcceleration() * timeStep);
	modelToControl->SetAngularVelocity(result);
}

void PhysicsController::CalculateLinearVelocity(Vector2DPtr originVector)
{
	// Calculate linear velocity from angular velocity
	// It calculates the linear velocity for a single point on the rotating body,
	// not the total linear velocity.
	
	// linear velocity = (angular velocity) * (perp vector from origin to the point)

	Vector2D perpVector;
	perpVector = Vector2D::GetPerpendicularVector(originVector);
    //perpVector = originVector->GetPerpendicularVector(originVector);
	
	double angularVelocityValue;
	angularVelocityValue = modelToControl->GetAngularVelocity();
	
	Vector2D result;
	result = Vector2D::ScalarProduct(&perpVector, angularVelocityValue);
    //result = result.ScalarProduct(&perpVector, angularVelocityValue);
	modelToControl->SetVelocity(result);
}

void PhysicsController::Rotate(double timeStep)
{
	// new orientation = old orientation + (angular velocity * time step)
	
	double result;
	
	result = modelToControl->GetOrientation() + (modelToControl->GetAngularVelocity() * timeStep);
	modelToControl->SetOrientation(result);
}


void PhysicsController::Stand(void)
{
	Vector2D velocityVector = modelToControl->GetVelocity();
	velocityVector.SetXComponent(kStandingSpeed);
	velocityVector.SetYComponent(kStandingSpeed);
	modelToControl->SetVelocity(velocityVector);
}


// Movement determination functions
Boolean PhysicsController::CanMoveUp(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;

	// Test if the tile distance pixels above ours is a wall
	short rowToTest = (modelToControl->GetWorldY() - distance) / kTileHeight;
	short columnToTest = modelToControl->GetWorldX() / kTileWidth;

	// Bounds checking
	if (rowToTest < 0)
		return false;
	
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;
		
	// We've just tested the left edge of the sprite.
	// Now test the right edge.
	short rightEdge = modelToControl->GetWorldX() + modelToControl->GetSpriteWidth();
	columnToTest = rightEdge / kTileWidth;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;
	
	// At this point, there are no walls blocking us
	return true;
}
 
Boolean PhysicsController::CanMoveDown(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.

	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;

	// Test if the tile distance pixels below ours is a wall
	short rowToTest = (modelToControl->GetWorldY() + 
		modelToControl->GetSpriteHeight() + distance) / kTileHeight;
	short columnToTest = modelToControl->GetWorldX() / kTileWidth;	

	// Bounds checking
	if (rowToTest >= currentLevel->GetLevelHeight())
		return false;
	
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;
		
	// We've just tested the left edge of the sprite.
	// Now test the right edge;
	short rightEdge = modelToControl->GetWorldX() + modelToControl->GetSpriteWidth();
	columnToTest = rightEdge / kTileWidth;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;
	
	// At this point, there are no walls blocking us
	return true;

}

Boolean PhysicsController::CanMoveLeft(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.

	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;

	// Test if the tile distance pixels to the left of ours is a wall
	short rowToTest = modelToControl->GetWorldY() / kTileHeight;
	short columnToTest = (modelToControl->GetWorldX() - distance) / kTileWidth;	

	// Bounds checking
	if (columnToTest < 0)
		return false;

	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();	
	if (tileAttribute == kWallTile) 
		return false;
		
	// We've just tested the top edge of the sprite.
	// Now test the bottom edge;
	short bottomEdge = modelToControl->GetWorldY() + modelToControl->GetSpriteHeight();
	rowToTest = bottomEdge / kTileHeight;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();	
	if (tileAttribute == kWallTile)
		return false;
	
	// At this point, there are no walls blocking us
	return true;

}

Boolean PhysicsController::CanMoveRight(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;

	// Test if the tile distance pixels to the right of ours is a wall
	short rowToTest = modelToControl->GetWorldY() / kTileHeight;
	short columnToTest = (modelToControl->GetWorldX() + 
		modelToControl->GetSpriteWidth() + distance) / kTileWidth;

	// Bounds checking
	if (columnToTest >= currentLevel->GetLevelWidth())
		return false;
			
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();	
	if (tileAttribute == kWallTile)
		return false;
		
	// We've just tested the top edge of the sprite.
	// Now test the bottom edge;
	short bottomEdge = modelToControl->GetWorldY() + modelToControl->GetSpriteHeight();
	rowToTest = bottomEdge / kTileHeight;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();	
	if (tileAttribute == kWallTile)
		return false;
	
	// At this point, there are no walls blocking us
	return true;
}

Boolean PhysicsController::CanMoveUpAndLeft(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	// Test if the tile one pixel above ours is a wall
	short rowToTest = (modelToControl->GetWorldY() - distance) / kTileHeight;
	short columnToTest = modelToControl->GetWorldX() / kTileWidth;

	// Bounds checking
	if (rowToTest < 0)
		return false;

	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one column to the left to see if the tile above
	// and to the left is a wall
	columnToTest--;

	// Bounds checking
	if (columnToTest < 0)
		return false;

	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one row down and test if it is a wall
	rowToTest++;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// At this point, none of the three tiles we tested is a wall.
	// We can move up and left because the 4th tile to test is part
	// of the current sprite position and we know that's not a wall.
	return true;
}

Boolean PhysicsController::CanMoveUpAndRight(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	// Test if the tile one pixel above the sprite's right edge is a wall
	short rowToTest = (modelToControl->GetWorldY() - distance) / kTileHeight;
	short columnToTest = (modelToControl->GetWorldX() + 
		modelToControl->GetSpriteWidth()) / kTileWidth;

	// Bounds checking
	if (rowToTest < 0)
		return false;
		
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one column to the right to see if the tile above
	// and to the right is a wall
	columnToTest++;
    
	// Bounds checking
	if (columnToTest >= currentLevel->GetLevelWidth())
		return false;

	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one row down and test if it is a wall
	rowToTest++;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// At this point, none of the three tiles we tested is a wall.
	// We can move up and left because the 4th tile to test is part
	// of the current sprite position and we know that's not a wall.
	return true;

}

Boolean PhysicsController::CanMoveDownAndLeft(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	// Test if the tile one pixel to the left of the bottom left corner is a wall
	short rowToTest = (modelToControl->GetWorldY() + 
		modelToControl->GetSpriteHeight()) / kTileHeight;
	short columnToTest = (modelToControl->GetWorldX() - distance) / kTileWidth;

	// Bounds checking
	if (columnToTest < 0)
		return false;
	
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one row down to see if the tile below
	// and to the left is a wall
	rowToTest++;
    
	// Bounds checking
	if (rowToTest >= currentLevel->GetLevelHeight())
		return false;

	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one column right and test if it is a wall
	columnToTest++;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// At this point, none of the three tiles we tested is a wall.
	// We can move up and left because the 4th tile to test is part
	// of the current sprite position and we know that's not a wall.
	return true;

}

Boolean PhysicsController::CanMoveDownAndRight(short distance, GameLevelPtr currentLevel, GameTileListPtr theTileList)
{
	// Distance is the number of pixels we want to move.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = currentLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	// Test if the tile one pixel below the bottom right corner is a wall
	short rowToTest = (modelToControl->GetWorldY() + 
		modelToControl->GetSpriteHeight() + distance) / kTileHeight;
		
	short columnToTest = (modelToControl->GetWorldX() + 
		modelToControl->GetSpriteWidth()) / kTileWidth;

	// Bounds checking
	if (rowToTest >= currentLevel->GetLevelHeight())
		return false;
        
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one column to the right to see if the tile below
	// and to the right is a wall
	columnToTest++;
    
	// Bounds checking
	if (columnToTest >= currentLevel->GetLevelWidth())
		return false;
    
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// Now move one row up and test if it is a wall
	rowToTest--;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = currentLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTileList->tileTable[tileNum].GetTileType();
	if (tileAttribute == kWallTile)
		return false;

	// At this point, none of the three tiles we tested is a wall.
	// We can move up and left because the 4th tile to test is part
	// of the current sprite position and we know that's not a wall.
	return true;

}

short PhysicsController::HowFarUp(void)
{
	// This function is called after CanMoveUp fails.  So we can assume
	// that the adjacent tile is a wall.  We just check how many pixels
	// y is from the next tile.
	
	short result;
	
	result = modelToControl->GetWorldY() % kTileHeight;
	
	return result;
}

short PhysicsController::HowFarDown(void)
{
	// This function is called after CanMoveDown fails.  So we can assume
	// that the adjacent tile is a wall.  We just check how many pixels
	// y is from the next tile.

	short result;
	short tempResult;
	
	// As an example, x = 96, which lines up on a tile boundary.  Y % kTileHeight
	// will be 0, which is correct.  If y = 101, y is 27 pixels from the next tile.
	// However, 101 % 32 will be 5.  So we take kTileHeight and subtract 5 to get 
	// the proper value of 27.
	
	// Without the +1, the wall is detected too early and the
	// player stops too far away from the wall.
	short bottomEdge = modelToControl->GetWorldY() + modelToControl->GetSpriteHeight() + 1;
	tempResult = bottomEdge % kTileHeight;
	if (tempResult == 0)
		result = tempResult;
	else
		// Without the -1, the character crosses a tile boundary.
		// This causes problems such as characters running through
		// 1 tile thick walls and characters getting embedded one pixel
		// into walls, making movement out of them impossible.
		result = kTileHeight - tempResult - 1;
		
	return result;

}

short PhysicsController::HowFarLeft(void)
{
	// This function is called after CanMoveLeft fails.  So we can assume
	// that the adjacent tile is a wall.  We just check how many pixels
	// x is from the next tile.

	short result;
	
	result = modelToControl->GetWorldX() % kTileWidth;
	
	return result;
	

}

short PhysicsController::HowFarRight(void)
{
	// This function is called after CanMoveRight fails.  So we can assume
	// that the adjacent tile is a wall.  We just check how many pixels
	// x is from the next tile.

	short result;
	short tempResult;
	
	// As an example, x = 96, which lines up on a tile boundary.  X % kTileWidth
	// will be 0, which is correct.  If x = 101, x is 27 pixels from the next tile.
	// However, 101 % 32 will be 5.  So we take kTileWidth and subtract 5 to get 
	// the proper value of 27.

	// Without the +1, the wall is detected too early and the
	// player stops too far away from the wall.
	short rightEdge = modelToControl->GetWorldX() + modelToControl->GetSpriteWidth() + 1;
	tempResult = rightEdge % kTileWidth;
	if (tempResult == 0)
		result = tempResult;
	else
		// Without the -1, the character crosses a tile boundary.
		// This causes problems such as characters running through
		// 1 tile thick walls and characters getting embedded one pixel
		// into walls, making movement out of them impossible.
		result = kTileWidth - tempResult - 1;
	
	return result;

}


// Collision related functions
Boolean PhysicsController::DetectCollision(GameSpritePtr collidingModel)
{	
	// Calculate the bounding boxes
	short topEdge = modelToControl->GetWorldY();
	short bottomEdge = topEdge + modelToControl->GetSpriteHeight();
	short leftEdge = modelToControl->GetWorldX();
	short rightEdge = leftEdge + modelToControl->GetSpriteWidth();
	
	short colliderTopEdge = collidingModel->GetWorldY();
	short colliderBottomEdge = colliderTopEdge + collidingModel->GetSpriteHeight();
	short colliderLeftEdge = collidingModel->GetWorldX();
	short colliderRightEdge = colliderLeftEdge + collidingModel->GetSpriteWidth();
	
	if (topEdge > colliderBottomEdge)
		return false;
		
	if (bottomEdge < colliderTopEdge)
		return false;
		
	if (leftEdge > colliderRightEdge)
		return false;
		
	if (rightEdge < colliderLeftEdge)
		return false;

	// At this point, the two sprite bounding boxes intersect.
	// Check to see if there actually is a collision
	return (DetectExactCollision(collidingModel)); 
}

Boolean PhysicsController::DetectExactCollision(GameSpritePtr collidingModel)
{
	Rect spriteRect;
	Rect collidingSpriteRect;
	Rect overlapRect;
	
	spriteRect.top = modelToControl->GetWorldY();
	spriteRect.bottom = spriteRect.top + modelToControl->GetSpriteHeight();
	spriteRect.left = modelToControl->GetWorldX();
	spriteRect.right = spriteRect.left + modelToControl->GetSpriteWidth();
	
	collidingSpriteRect.top = collidingModel->GetWorldY();
	collidingSpriteRect.bottom = collidingSpriteRect.top + collidingModel->GetSpriteHeight();
	collidingSpriteRect.left = collidingModel->GetWorldX();
	collidingSpriteRect.right = collidingSpriteRect.left + collidingModel->GetSpriteWidth();
	
	Boolean haveOverlap;
	
	haveOverlap = SectRect(&spriteRect, &collidingSpriteRect, &overlapRect);
	if (haveOverlap == false)
		return false;
		
	// Go through all the pixels in the rectangle that overlaps
	// the two sprites.  If one of the pixels in both sprites is not
	// transparent then there is a collision.

	short spritePixelRow;
	short spritePixelColumn;
	short spriteTestRow;
	short spriteTestColumn;
	short spritePixelValue;
	GameOffscreenBufferPtr theSpriteStorage;
	
	short collidingSpritePixelRow;
	short collidingSpritePixelColumn;
	short collidingSpriteTestRow;
	short collidingSpriteTestColumn;
	short collidingSpritePixelValue;
	GameOffscreenBufferPtr collidingSpriteStorage;

	Rect localOverlapRect;
	
	localOverlapRect.top = 0;
	localOverlapRect.bottom = overlapRect.bottom - overlapRect.top;
	localOverlapRect.left = 0;
	localOverlapRect.right = overlapRect.right - overlapRect.left;

	for (short row = localOverlapRect.top; row < localOverlapRect.bottom; row++) {
		for (short column = localOverlapRect.left; column < localOverlapRect.right; column++) {
			// Get the sprite's local pixel.
			spritePixelRow = overlapRect.top - spriteRect.top + row;
			spritePixelColumn = overlapRect.left - spriteRect.left + column;
			
			// Get the pixel to test in the sprite buffer
			spriteTestRow = (modelToControl->GetSpriteRow() * 
					modelToControl->GetSpriteHeight()) + spritePixelRow;
			spriteTestColumn = (modelToControl->GetSpriteColumn() * 
					modelToControl->GetSpriteWidth()) + spritePixelColumn;
			
			theSpriteStorage = modelToControl->GetSpriteStorage();
			spritePixelValue = theSpriteStorage->GetPixelValue(spriteTestRow, spriteTestColumn);
			
			// Repeat the process for the colliding sprite
			collidingSpritePixelRow = overlapRect.top - collidingSpriteRect.top + row;
			collidingSpritePixelColumn = overlapRect.left - collidingSpriteRect.left + column;
			
			collidingSpriteTestRow = (collidingModel->GetSpriteRow() * 
					collidingModel->GetSpriteHeight()) + collidingSpritePixelRow;
			collidingSpriteTestColumn = (collidingModel->GetSpriteColumn() *
					collidingModel->GetSpriteWidth()) + collidingSpritePixelColumn;
				
			collidingSpriteStorage = collidingModel->GetSpriteStorage();
			collidingSpritePixelValue = collidingSpriteStorage->
					GetPixelValue(collidingSpriteTestRow, collidingSpriteTestColumn);
				 
			// Check if the two sprites have transparent values
			// at their current pixels.
			if ((spritePixelValue != kSpriteTransparencyValue) &&
				(collidingSpritePixelValue != kSpriteTransparencyValue))
					return true;
					
		}
	}
	
	// At this point, there was no collision
	return false;
}

void PhysicsController::ResolveCollision(GameCollisionPtr theCollision)
{
	// In this case, I don't bother with angular effects.
	// If you want angular effects, call CalculateAngularImpulse()
	// and CalculatePostCollisionAngularVelocities() here.
	double impulse = CalculateLinearImpulse(theCollision);
			
	CalculatePostCollisionLinearVelocities(theCollision, impulse);
}

double PhysicsController::CalculateLinearImpulse(GameCollisionPtr theCollision)
{
	GameSpritePtr objectA = theCollision->GetFirstObject();
	GameSpritePtr objectB = theCollision->GetSecondObject();

	// Calculate relative velocity
	Vector2D objectAVelocity = objectA->GetVelocity();
	Vector2D objectBVelocity = objectB->GetVelocity();	
	Vector2D relativeVelocity = Vector2D::VectorDifference(&objectAVelocity, &objectBVelocity);
	
	// Calculate numerator
	// Dot Product [(-1 + e) * (relative velocity), (Normal vector)]
	 
	Vector2D normalVector = theCollision->GetCollisionNormal();
	Vector2D relativeVelocityProduct = Vector2D::ScalarProduct(&relativeVelocity,
			(-1.0 * theCollision->GetCoefficientOfRestitution()));
	double impulseNumerator = Vector2D::DotProduct(&relativeVelocityProduct, &normalVector);
			
	// Calculate (1/Ma) + (1/Mb)
	double oneOverMass = (1.0 / objectA->GetMass()) + (1.0 / objectB->GetMass());
			
	// Multiply it by the normal vector
	// n * [(1/Ma) + (1/Mb)]
	Vector2D normalTimesOneOverMass = Vector2D::ScalarProduct(&normalVector, oneOverMass);
	
	// Build the denominator
	// DotProduct [n, (n * ((1/Ma) + (1/Mb))] 
	double impulseDenominator = Vector2D::DotProduct(&normalVector, 
			&normalTimesOneOverMass);
	
	double result = impulseNumerator / impulseDenominator;
	
	return result;
}

double PhysicsController::CalculateAngularImpulse(GameCollisionPtr theCollision)
{
	GameSpritePtr objectA = theCollision->GetFirstObject();
	GameSpritePtr objectB = theCollision->GetSecondObject();
	
	Vector2D objectAVelocity = objectA->GetVelocity();
	Vector2D objectBVelocity = objectB->GetVelocity();	
	Vector2D relativeVelocity = Vector2D::VectorDifference(&objectAVelocity, &objectBVelocity);
	
	Vector2D normalVector = theCollision->GetCollisionNormal();
	
	// Calculate the portion of the impulse that is identical
	// to the linear impulse
	Vector2D relativeVelocityProduct = Vector2D::ScalarProduct(&relativeVelocity,
			(-1.0 * theCollision->GetCoefficientOfRestitution()));
	double impulseNumerator = Vector2D::DotProduct(&relativeVelocityProduct, &normalVector);

	double oneOverMass = (1.0 / objectA->GetMass()) + (1.0 / objectB->GetMass());
			
	Vector2D normalTimesOneOverMass = Vector2D::ScalarProduct(&normalVector, 
			oneOverMass);
	
	double linearImpulseDenominator = Vector2D::DotProduct(&normalVector, 
			&normalTimesOneOverMass);
	
	// Calculate first part of angular impulse demoninatior
	// (perp vector dot n)^2 / moment of inertia A
	Vector2D centerOfMassAVector = theCollision->GetCollisionPointToCenterOfMassA();
	Vector2D CenterOfMassToCornerPerpA = Vector2D::GetPerpendicularVector(&centerOfMassAVector);
	
	double perpDotA = Vector2D::DotProduct(&CenterOfMassToCornerPerpA, &normalVector);
	double momentOfInertiaA = CalculateMomentOfInertia(objectA);
	double angularA = perpDotA * perpDotA / momentOfInertiaA;
				
	// Calculate second part of angular impulse denominator
	// (perp vector dot n)^2 / moment of inertia B
	Vector2D centerOfMassBVector = theCollision->GetCollisionPointToCenterOfMassB();
	Vector2D CenterOfMassToCornerPerpB = Vector2D::GetPerpendicularVector(&centerOfMassAVector);

	double perpDotB = Vector2D::DotProduct(&CenterOfMassToCornerPerpB, &normalVector);
	double momentOfInertiaB = CalculateMomentOfInertia(objectB);
	double angularB = perpDotB * perpDotB / momentOfInertiaB;
	
	// Make final calculations
	double angularImpulseDenominator = angularA + angularB;
	double impulseDenominator = linearImpulseDenominator + angularImpulseDenominator;
	double result = impulseNumerator / impulseDenominator;
	
	return result;
}	

void PhysicsController::CalculatePostCollisionLinearVelocities(GameCollisionPtr theCollision, double impulse)
{
	// Calculate the first object's outgoing velocity
	// new velocity = (old velocity) + [(normal vector) * impulse / mass]
	GameSpritePtr objectA = theCollision->GetFirstObject();
	Vector2D velocityVectorA = objectA->GetVelocity();

	Vector2D normalVector = theCollision->GetCollisionNormal();
	
	double impulseOverMassA = impulse / objectA->GetMass();
	Vector2D offsetA = Vector2D::ScalarProduct(&normalVector, impulseOverMassA);
	velocityVectorA = Vector2D::VectorSum(&velocityVectorA, &offsetA);
	objectA->SetVelocity(velocityVectorA);
	
	// Calculate second object's outgoing velocity
	// new velocity = (old velocity) - [(normal vector) * impulse / mass]
	GameSpritePtr objectB = theCollision->GetSecondObject();
	Vector2D velocityVectorB = objectB->GetVelocity();
	
	double impulseOverMassB = (impulse / objectB->GetMass()) * -1.0;
	Vector2D offsetB = Vector2D::ScalarProduct(&normalVector, impulseOverMassB);
	velocityVectorB = Vector2D::VectorSum(&velocityVectorB, &offsetB);
	objectA->SetVelocity(velocityVectorB);

}
	
void PhysicsController::CalculatePostCollisionAngularVelocities(GameCollisionPtr theCollision, double impulse)
{
	Vector2D normalVector = theCollision->GetCollisionNormal();
	
	// Calculate first object's outgoing angular velocity
	// new velocity = (old velocity) + (Dot Product (perp vector, jn) / moment of inertia)
	GameSpritePtr objectA = theCollision->GetFirstObject();
	Vector2D jn = Vector2D::ScalarProduct(&normalVector, impulse);
	
	Vector2D centerOfMassAVector = theCollision->GetCollisionPointToCenterOfMassA();
	Vector2D centerOfMassToCornerPerpA = Vector2D::GetPerpendicularVector(&centerOfMassAVector);

	double numeratorA = Vector2D::DotProduct(&centerOfMassToCornerPerpA, &jn);
	double momentOfInertiaA = CalculateMomentOfInertia(objectA);
		
	double angularVelocityValueA = objectA->GetAngularVelocity();
	angularVelocityValueA = angularVelocityValueA + (numeratorA / momentOfInertiaA);
	
	objectA->SetAngularVelocity(angularVelocityValueA);
	
	// Calculate second object's outgoing angular velocity
	// new velocity = (old velocity) + (Dot Product (perp vector, -jn) / moment of inertia)
	GameSpritePtr objectB = theCollision->GetSecondObject();
	double angularVelocityB = objectB->GetAngularVelocity();
	
	Vector2D negativeJN = Vector2D::ScalarProduct(&normalVector, (impulse * -1.0));

	Vector2D centerOfMassBVector = theCollision->GetCollisionPointToCenterOfMassB();
	Vector2D centerOfMassToCornerPerpB = Vector2D::GetPerpendicularVector(&centerOfMassBVector);
			
	double numeratorB = Vector2D::DotProduct(&centerOfMassToCornerPerpB, &negativeJN);
	double momentOfInertiaB = CalculateMomentOfInertia(objectB);
	
	angularVelocityB= angularVelocityB + (numeratorB / momentOfInertiaB);
	objectB->SetAngularVelocity(angularVelocityB);
}	
	
