// InputController.cp		Written by Mark Szymczyk
// Book version

// Make sure InputController.h is included just once
#ifndef INPUT_CONTROLLER
	#include "InputController.h"
#endif

InputController::InputController(void)
{
    masterPort = nil;
    eventList = nil;
	controllerDeviceInterface = nil;
    controllerDevice = nil;
}

InputController::~InputController(void)
{
	CleanUp();
}


// Accessors
short InputController::GetXAxisValue(void) 
{ 
    return xAxisValue; 
}

void InputController::SetXAxisValue(short x) 
{ 
    xAxisValue = x; 
}

short InputController::GetYAxisValue(void) 
{ 
    return yAxisValue; 
}

void InputController::SetYAxisValue(short y) 
{ 
    yAxisValue = y; 
}

mach_port_t InputController::GetMasterPort(void)
{
    return masterPort;
}

void InputController::SetMasterPort(mach_port_t thePort)
{
    masterPort = thePort;
}

LinkedList InputController::GetElementList(void)
{
    return elementList;
}

void InputController::SetElementList(LinkedList theList)
{
    elementList = theList;
}

LinkedList InputController::GetControllerList(void)
{
    return controllerList;
}

void InputController::SetControllerList(LinkedList theList)
{
    controllerList = theList;
}

InputDevicePtr InputController::GetControllerDevice(void)
{
    return controllerDevice;
}

void InputController::SetControllerDevice(InputDevicePtr theDevice)
{
    controllerDevice = theDevice;
}
        
IOHIDDeviceInterfaceHandle InputController::GetControllerDeviceInterface(void)
{
    return controllerDeviceInterface;
}
        
void InputController::SetControllerDeviceInterface(IOHIDDeviceInterfaceHandle theInterface)
{
    controllerDeviceInterface = theInterface;
}

IOHIDQueueInterfaceHandle InputController::GetEventList(void)
{
    return eventList;
}
        
void InputController::SetEventList(IOHIDQueueInterfaceHandle theList)
{
    eventList = theList;
}

// Setup functions
void InputController::Initialize(void)
{
    CreateMasterPort();
    SetDefaultController();
    SetDefaultControls();    
    //CreateControllerList();    
}

void InputController::Setup(void)
{
    CreateDeviceInterface();
    OpenDevice();
    CreateEventList();
}

// HID Manager initialization functions
void InputController::CreateMasterPort(void)
{
    IOReturn result;
    result = IOMasterPort(bootstrap_port, &masterPort);
}

void InputController::CreateControllerList(void)
{
    IOReturn result;
    CFMutableDictionaryRef matchingDictionary;
    Boolean noMatchingDevices;
    io_iterator_t deviceList;

    // Dispose of any previously created controller list
    DisposeControllerList();
    
    // Create matching dictionary of HID devices
    matchingDictionary = IOServiceMatching(kIOHIDDeviceKey);
    
    // Look for any HID controllers connected
    // to the player's machine
    result = IOServiceGetMatchingServices(masterPort, matchingDictionary, &deviceList);
    
    if ((deviceList == nil) || (result != kIOReturnSuccess))
        noMatchingDevices = true;
    else
        noMatchingDevices = false;

    if (noMatchingDevices) {
        // In future chapters, we will use the keyboard
        // if the player does not have any HID Manager
        // devices connected to his machine.
        return;
        //ExitToShell();
    }
    else {
        FillControllerList(deviceList);
    }
    
    // We're finished with the matching dictionary.
    // IOServiceGetMatchingServices() released it for us.
    matchingDictionary = nil;
    
    IOObjectRelease(deviceList);
}

void InputController::FillControllerList(io_iterator_t deviceList)
{
    io_object_t currentDevice;
    kern_return_t result;
    CFMutableDictionaryRef deviceProperties;
    InputDevicePtr deviceToAdd = new InputDevice;
    
    currentDevice = IOIteratorNext(deviceList);

    while (currentDevice != nil) {
        // Retrieve the name of the current controller
        result = IORegistryEntryCreateCFProperties(currentDevice, &deviceProperties,
                       		 kCFAllocatorDefault, kNilOptions);
        
        // Fill our InputDevice object with its data
        // and add it to the controller list.
        deviceToAdd->SetDeviceObject(currentDevice);
        deviceToAdd->StoreDeviceData(deviceProperties);
        controllerList.AddItem(deviceToAdd);
        
        // Get next controller
        deviceToAdd = new InputDevice;
        currentDevice = IOIteratorNext(deviceList);
    }
    
}

void InputController::CreateElementList(io_object_t device)
{
    // Gets a list of elements (controls)
    // for an input device.
    
    // Dispose of any previously created element list
    DisposeElementList();
    
    kern_return_t result;
    CFMutableDictionaryRef deviceProperties;

    result = IORegistryEntryCreateCFProperties(device, &deviceProperties,
                kCFAllocatorDefault, kNilOptions);
                
    if (result != KERN_SUCCESS)
        return; //elementList;
     
    // Get first element
    CFTypeRef firstElement = CFDictionaryGetValue(deviceProperties, CFSTR(kIOHIDElementKey));
    
    CFTypeID elementType = CFGetTypeID(firstElement);
    if (elementType == CFArrayGetTypeID()) {
        FillElementList(firstElement);
    }

}

void InputController::FillElementList(CFTypeRef currentElement)
{
    // Lists of HID elements are stored in nested arrays.
    // Mac OS X's Core Foundation arrays let you apply a function
    // to each element of an array by calling CFApplyFunction().
    // This is what I do to fill the element list.
    
    CFTypeID currentElementType = CFGetTypeID(currentElement);
    CFTypeID arrayType = CFArrayGetTypeID();
    
    if (currentElementType == arrayType) {
        CFIndex elementCount = CFArrayGetCount((CFArrayRef)currentElement);
        CFRange elementRange = {0, elementCount};    
        CFArrayApplyFunction((CFArrayRef)currentElement, elementRange, 
                InputController::AddElementToList, (void *)this);
    }
}

void InputController::AddElementToList(const void* value, void* parameter)
{
    // The value parameter contains the Core Foundation element data.
    // The parameter parameter contains our InputController object.
    
    CFTypeRef elementData = (CFTypeRef) value;
    InputControllerPtr currentController = (InputControllerPtr) parameter;
    LinkedList theElementList = currentController->GetElementList();
    
    // Store the element's data.
    InputDeviceElementPtr elementToAdd = new InputDeviceElement;
    elementToAdd->StoreElementData(elementData);
    
    // If the element is a collection of other elements, call
    // FillElementList() to fill the elements in the collection.
    // Otherwise, add the element to the list.
    if (elementToAdd->GetType() == kIOHIDElementTypeCollection) {
        CFTypeRef collectionElement = CFDictionaryGetValue((CFDictionaryRef)elementData, 
                CFSTR(kIOHIDElementKey));
        currentController->FillElementList(collectionElement);
    }
    else {
        theElementList.AddItem(elementToAdd);
        currentController->SetElementList(theElementList);
    }
    
}

void InputController::CreateDeviceInterface(void)
{
    IOReturn result;
    io_name_t deviceClassName;

    if (controllerDevice == nil)
        return;
        
    result = IOObjectGetClass(controllerDevice->GetDeviceObject(), deviceClassName);
    
    IOCFPlugInInterfaceHandle plugInInterface;
    HRESULT plugInResult;
    SInt32 score;
    
    // Create the plugin so we can create the device interface.
    result = IOCreatePlugInInterfaceForService (controllerDevice->GetDeviceObject(),
                kIOHIDDeviceUserClientTypeID, kIOCFPlugInInterfaceID,
                &plugInInterface, &score);
                
    // Create the device interface
    plugInResult = (*plugInInterface)->QueryInterface((IOCFPlugInInterface*)plugInInterface,
                        CFUUIDGetUUIDBytes(kIOHIDDeviceInterfaceID),
                        (LPVOID*)&controllerDeviceInterface);
                        
    // Free up the plugin
    (*plugInInterface)->Release(plugInInterface);
}

void InputController::OpenDevice(void)
{
    IOReturn result;
    
    if (controllerDeviceInterface == nil)
        return;
        
    result = (*controllerDeviceInterface)->open(controllerDeviceInterface, 0);
}

void InputController::CreateEventList(void)
{
    if (controllerDeviceInterface == nil)
        return;
        
    // Allocate space for the event list.
    eventList = (*controllerDeviceInterface)->allocQueue(controllerDeviceInterface);
    
    if (eventList == nil)
        return;
        
    IOReturn result;
    
    // Create the event list.
    result = (*eventList)->create(eventList, 0, kEventListSize);
    if (result != kIOReturnSuccess)
        return;
     
    Pause();
    // Add button elements to the queue
    result = (*eventList)->addElement(eventList,
                            gameNeeds[kAttackNeed], 0);
    if (result != kIOReturnSuccess)
        return;

    result = (*eventList)->addElement(eventList,
                            gameNeeds[kChangeWeaponNeed], 0);
                            
    if (result != kIOReturnSuccess)
        return;

    // I don't use the pause game need because the HID
    // Manager does not support the mouse and keyboard.
    // If you want to let the player pause the game with
    // one of the gamepad buttons, remove the comments.
    /*
    result = (*eventList)->addElement(eventList,
                                gameNeeds[kPauseGameNeed], 0);
    */
    
}


// Setting the default controls
void InputController::SetDefaultController(void)
{
    // This function finds the first joystick or gamepad
    // connected to the player's machine and uses that as
    // the default controller.  This function allows the
    // player to play the game without having to configure
    // the controllers first.
    
    CreateControllerList();
    Iterator controllerListIterator(&controllerList);
    InputDevicePtr currentInputDevice;

    long primaryUsage;
    long primaryUsagePage;

    currentInputDevice = (InputDevicePtr)controllerListIterator.Next();
    
    while(currentInputDevice != nil) {
        // Find primary usage of this controller
        primaryUsagePage = currentInputDevice->GetUsagePage();
        primaryUsage = currentInputDevice->GetUsage();
        
        // If it's a joystick or gamepad, make it the default controller
        if(primaryUsagePage == kGenericDesktopPage) {
            if (primaryUsage == kJoystickUsage) {
                SetControllerDevice(currentInputDevice);
                return;
            }
            else if (primaryUsage == kGamepadUsage) {
                SetControllerDevice(currentInputDevice);
                return;
            }
        }
        
        // Get next controller in list
        currentInputDevice = (InputDevicePtr)controllerListIterator.Next();
    }
    
}

void InputController::SetDefaultControls(void)
{
    // This function assigns default controls so the player
    // can play without configuring the controls.  It will use
    // the x-axis and y-axis elements for movement, the first button 
    // for attacking, and the second button for changing weapons.
    
    if (controllerDevice == nil)
        return;
        
    io_object_t deviceObj = controllerDevice->GetDeviceObject();
    CreateElementList(deviceObj);
    
    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    long primaryUsage;
    long primaryUsagePage;

    int elementCount = elementList.GetNumberOfItems();
    
    while(elementCount > 0) {
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        // Find primary usage of this controller
        primaryUsagePage = currentInputElement->GetUsagePage();
        primaryUsage = currentInputElement->GetUsage();
        
        if (primaryUsagePage == kGenericDesktopPage) {
            // If the element is primarily used for axis movement
            // assign it to axis movement by default.
            if (primaryUsage == kXAxisUsage) {
                gameNeeds[kXAxisNeed] = currentInputElement->GetCookie();
            }
            else if (primaryUsage == kYAxisUsage) {
                gameNeeds[kYAxisNeed] = currentInputElement->GetCookie();
            }
        }
        // Make button 1 the default attack button
        // and button 2 the default change weapon button
        else if (primaryUsagePage == kButtonPage) {
            if (primaryUsage == kButton1Usage) {
                gameNeeds[kAttackNeed] = currentInputElement->GetCookie();
            }
            else if (primaryUsage == kButton2Usage) {
                gameNeeds[kChangeWeaponNeed] = currentInputElement->GetCookie();
            }
        }
        
        // Get next element
        elementCount--;
    }
}


// Clean up functions
void InputController::CleanUp(void)
{
    DisposeEventList();
    CloseDevice();
    DisposeDeviceInterface();
    DisposeElementList();
    DisposeControllerList();
    DisposeMasterPort();
}

void InputController::DisposeEventList(void)
{
    IOReturn result;
    
    if (eventList != nil) {
        result = (*eventList)->dispose(eventList);
        (*eventList)->Release(eventList);
        eventList = nil;
    }
}

void InputController::CloseDevice(void)
{
    IOReturn result;
    
    if (controllerDeviceInterface != nil) {
        result = (*controllerDeviceInterface)->close(controllerDeviceInterface);
    }
}

void InputController::DisposeDeviceInterface(void)
{
    if (controllerDeviceInterface != nil) {
        (*controllerDeviceInterface)->Release(controllerDeviceInterface);
        controllerDeviceInterface = nil;
    }
    
}

void InputController::DisposeElementList(void)
{
    Iterator elementListIterator(&elementList);
    InputDevicePtr currentInputDeviceElement;

    int itemsToDispose = elementList.GetNumberOfItems();
    
    while(itemsToDispose > 0) {        
        currentInputDeviceElement = (InputDevicePtr)elementListIterator.Next();
        elementList.RemoveItem(currentInputDeviceElement);
        itemsToDispose--;
    }
    
}

void InputController::DisposeControllerList(void)
{
    Iterator controllerListIterator(&controllerList);
    InputDevicePtr currentInputDevice;

    int itemsToDispose = controllerList.GetNumberOfItems();
    
    while(itemsToDispose > 0) {        
        currentInputDevice = (InputDevicePtr)controllerListIterator.Next();
        controllerList.RemoveItem(currentInputDevice);
        itemsToDispose--;
    }
    
}

void InputController::DisposeMasterPort(void)
{
    if (masterPort != nil)
        mach_port_deallocate(mach_task_self(), masterPort);
}

InputControllerAction InputController::DetermineAction(void)
{	
	// This commented out section of code reads button
    // presses without an event queue.  If you don't want
    // to use a queue, remove the comments from this section
    // and comment out the rest of the DetermineAction() function.
    /*
    HRESULT error;
			
    IOHIDEventStruct theEvent;
    
    if (controllerDeviceInterface == nil)
        return kNoButtonsPressed;
        
    // Check if the player pressed the attack button
    error = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kAttackNeed], &theEvent);
    if (theEvent.value == kButtonDown)
        return kAttack;
        
    // Check if the player pressed the change weapon button
    error = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kChangeWeaponNeed], &theEvent);
    if (theEvent.value == kButtonDown)
        return kChangeWeapon;

    // Place your other button needs here.
    
	// At this point, nothing occurred
	return kNoButtonsPressed;
    */
    
	HRESULT error;
    IOHIDEventStruct theEvent;
    
    if (eventList == nil)
        return kNoButtonsPressed;
        
    // Read next event in the list
    error = (*eventList)->getNextEvent(eventList, &theEvent, 
                    kZeroTime, 0);
    
    // Does the event list have no events in it?
    if (error == kIOReturnUnderrun)
        return kNoButtonsPressed;

    // Check each of the needs in the queue.
    if (theEvent.elementCookie == gameNeeds[kAttackNeed]) {
        if (theEvent.value == kButtonDown) {
            return kAttack;
        }
    }
    else if (theEvent.elementCookie == gameNeeds[kChangeWeaponNeed]) {
        if (theEvent.value == kButtonDown) {
            return kChangeWeapon;
        }
    }
    // Remove the comments if you want to support the
    // player pausing the game with a joystick button.
    /*
    else if (theEvent.elementCookie == gameNeeds[kPauseGameNeed]) {
        if (theEvent.value == kButtonDown) {
            return kPauseGame;
        }
    }
    */    
	// At this point, nothing occurred
	return kNoButtonsPressed;
}

InputControllerAction InputController::DetermineDigitalMovement(void)
{
    // I use analog movement in the HID Manager code in this book.
    // This code is here if you want to use digital movement in your
    // game.  I recommend going with analog movement.
    
    HRESULT result;
    IOHIDEventStruct event;

    Boolean movedUp;
    Boolean movedDown;
    Boolean movedLeft;
    Boolean movedRight;

    if (controllerDeviceInterface == nil)
        return kNoMovement;
        
    // Check for movement in the four directions
    // Up
    result = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kDirectionPadUpNeed], &event);
    if (event.value == kButtonDown)
        movedUp = true;
    else
        movedUp = false;
        
    // Down
    result = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kDirectionPadDownNeed], &event);
    if (event.value == kButtonDown)
        movedDown = true;
    else
        movedDown = false;

    // Left
    result = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kDirectionPadLeftNeed], &event);
    if (event.value == kButtonDown)
        movedLeft = true;
    else
        movedLeft = false;

    // Right
    result = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kDirectionPadRightNeed], &event);
    if (event.value == kButtonDown)
        movedRight = true;
    else
        movedRight = false;

    // Determine direction of movement
	if ((movedUp) && (movedLeft))
		return kMoveUpAndLeft;
	else if ((movedUp) && (movedRight))
		return kMoveUpAndRight;
	else if ((movedDown) && (movedLeft))
		return kMoveDownAndLeft;
	else if ((movedDown) && (movedRight))
		return kMoveDownAndRight;

	// At this point, we know the movement is not diagonal
	else if (movedUp)
		return kMoveUp;
	else if (movedDown)
		return kMoveDown;
	else if (movedLeft)
		return kMoveLeft;
	else if (movedRight) 
		return kMoveRight;
	else	
		return kNoMovement;

}

InputControllerAction InputController::DetermineAnalogMovement(void)
{
	// Determine if there was horizontal movement.  
	UInt32 xAxisInput;
    UInt32 yAxisInput;
	HRESULT result;
    IOHIDEventStruct event;

    if (controllerDeviceInterface == nil)
        return kNoMovement;
	
    // Check for horizontal movement
    result = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kXAxisNeed], &event);
                    
	if (result != noErr)
		return kNoMovement;
        
    xAxisInput = event.value;
    xAxisValue = ConvertXAxisData(xAxisInput, kAxisMinimumValue, kAxisMaximumValue);

    // Check for vertical movement
    result = (*controllerDeviceInterface)->getElementValue(controllerDeviceInterface,
                    gameNeeds[kYAxisNeed], &event);
                    
	if (result != noErr)
		return kNoMovement;
        
    yAxisInput = event.value;
    yAxisValue = ConvertYAxisData(yAxisInput, kAxisMinimumValue, kAxisMaximumValue);
	
    // Determine direction of movement
    if ((yAxisValue < kAxisCenterValue) && (xAxisValue == kAxisCenterValue))
		return kMoveUp;
	else if ((yAxisValue < kAxisCenterValue) && (xAxisValue < kAxisCenterValue))
        return kMoveUpAndLeft;
	else if ((yAxisValue < kAxisCenterValue) && (xAxisValue > kAxisCenterValue))
        return kMoveUpAndRight;
	else if ((yAxisValue > kAxisCenterValue) && (xAxisValue < kAxisCenterValue))
        return kMoveDownAndLeft;
	else if ((yAxisValue > kAxisCenterValue) && (xAxisValue > kAxisCenterValue))
        return kMoveDownAndRight;
	else if ((yAxisValue > kAxisCenterValue) && (xAxisValue == kAxisCenterValue))
		return kMoveDown;
	else if ((yAxisValue == kAxisCenterValue) && (xAxisValue < kAxisCenterValue))
		return kMoveLeft;
	else if ((yAxisValue == kAxisCenterValue) && (xAxisValue > kAxisCenterValue))
		return kMoveRight;
	else
		return kNoMovement;
	
}


void InputController::Configure(void)
{
    // Our configuration dialog box is movable modal so
    // we must show the dialog in the GameApp class,
    // which means this function does nothing.
    // If you were to create a modal dialog, you could
    // display it here and call the Toolbox function
    // ModalDialog() to handle events
}

void InputController::Pause(void)
{
	IOReturn result;
    
    // Stop receiving events in the event queue
    result = (*eventList)->stop(eventList);
}

void InputController::Resume(void)
{
	IOReturn result;
    
    // Start receiving events
    result = (*eventList)->start(eventList);
	
}

// Utility functions
Boolean InputController::AnyConnectedHIDDevices(void)
{
    IOReturn result;
    CFMutableDictionaryRef matchingDictionary;
    io_iterator_t deviceList;
    mach_port_t theMachPort;
    
    // Create the mach port
    result = IOMasterPort(bootstrap_port, &theMachPort);
    if (result != kIOReturnSuccess)
        return false;
        
    // Create matching dictionary of HID devices
    matchingDictionary = IOServiceMatching(kIOHIDDeviceKey);
    
    // Look for any HID controllers connected
    // to the player's machine
    result = IOServiceGetMatchingServices(theMachPort, matchingDictionary, &deviceList);
    
    Boolean matchingDevices;
    
    if ((deviceList == nil) || (result != kIOReturnSuccess))
        matchingDevices = false;
    else
        matchingDevices = true;
        
    // We're finished with the matching dictionary.
    // IOServiceGetMatchingServices() released it for us.
    matchingDictionary = nil;
    
    IOObjectRelease(deviceList);

    mach_port_deallocate(mach_task_self(), theMachPort);
    
    return matchingDevices;
}

SInt32 InputController::ConvertXAxisData(UInt32 axisValue, SInt32 minValue, SInt32 maxValue) 
{
	long axisMaximum = GetMaximumXAxisValue();
    SInt32 divisor = axisMaximum / (maxValue - minValue);
    
    // Check for division by 0
    if (divisor == 0)
        return kAxisCenterValue;
        
	SInt32 result = (axisValue / divisor) + minValue;
	return result;
}

SInt32 InputController::ConvertYAxisData(UInt32 axisValue, SInt32 minValue, SInt32 maxValue) 
{
	long axisMaximum = GetMaximumYAxisValue();	
    SInt32 divisor = axisMaximum / (maxValue - minValue);

    // Check for division by 0
    if (divisor == 0)
        return kAxisCenterValue;

	SInt32 result = (axisValue / divisor) + minValue;

    // We don't have to change the sign of result with the HID Manager.
    // The HID Manager returns lower values when the joystick is moved up
    // and higher values when the joystick is moved down, 
    // which is what we want.
	return result;
}

long InputController::GetMaximumXAxisValue(void)
{
    // Finds the maximum value for the element
    // the player is using for horizontal movement.
    
    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    IOHIDElementCookie cookieValue;

    int elementCount = elementList.GetNumberOfItems();
    
    while(elementCount > 0) {
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        // Find the element's cookie
        cookieValue = currentInputElement->GetCookie();
        
        // If this is the right element, return its raw maximum value.
        if (gameNeeds[kXAxisNeed] == cookieValue) {
            return (currentInputElement->GetRawMax());
        }
        
        // Get next element
        elementCount--;
    }
    
    return 0;
}

long InputController::GetMaximumYAxisValue(void)
{
    // Finds the maximum value for the element
    // the player is using for vertical movement.

    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    IOHIDElementCookie cookieValue;

    int elementCount = elementList.GetNumberOfItems();
    
    while(elementCount > 0) {
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        // Find the element's cookie
        cookieValue = currentInputElement->GetCookie();
        
        // If this is the right element, return its raw maximum value.
        if (gameNeeds[kYAxisNeed] == cookieValue) {
            return (currentInputElement->GetRawMax());
        }
        
        // Get next element
        elementCount--;
    }
    
    return 0;
}
