// GameSound.cp		By Mark Szymczyk

#ifndef GAME_SOUND
	#include "GameSound.h"
#endif

// Constructor
GameSound::GameSound(void)
{
	channel = nil;
	soundData = nil;
	sampleRate = 0;
	soundFileRefNum = kFileNotOpen;
	soundHeader = nil;
    buffer1 = nil;
    buffer2 = nil;
	startOfSoundData = 0;
}

// Destructor
GameSound::~GameSound(void)
{
	Dispose();
}


// Accessors
SndChannelPtr GameSound::GetChannel(void) 				
{ 
	return channel; 
}

void GameSound::SetChannel(SndChannelPtr theChannel)	
{ 
	channel = theChannel; 
}

SndListHandle GameSound::GetSoundData(void) 
{ 
	return soundData; 
}

Boolean GameSound::IsSoundPlaying(void)
{
    SCStatus  theStatus;
    OSErr error;

    error = SndChannelStatus(channel, sizeof(SCStatus), &theStatus);
    if (error == noErr) {
        return(theStatus.scChannelBusy);
    }
    return(false);
}


void GameSound::Create(void)	
{
	// Creates a sound channel with no callback routine
	
	OSErr error;
	
	channel = (SndChannelPtr)NewPtr(sizeof(SndChannel));
	channel->qLength = stdQLength;
	
	error = SndNewChannel(&channel, sampledSynth, initStereo, nil);
	
}

void GameSound::CreateWithCallback(SndCallBackProcPtr theCallback)	
{
	// Creates a sound channel with a callback routine. 
	// You supply the name of your callback routine as
	// the parameter to the function.  I include callback
	// routines to loop a sound and dispose of a sound.
	// You can add your own callback routines to the
	// GameSound class.
	
	OSErr error;
	SndCallBackUPP callbackUPP;
	
	channel = (SndChannelPtr)NewPtr(sizeof(SndChannel));
	channel->qLength = stdQLength;
	
	callbackUPP = NewSndCallBackUPP(theCallback);
	error = SndNewChannel(&channel, sampledSynth, initStereo, callbackUPP);
	
}

void GameSound::ReadSoundData(short soundID)
{
	if (soundData != nil)
		DisposeSoundData();
		
	soundData = (SndListHandle)GetResource('snd ', soundID);
}


// Play sounds
void GameSound::Play(void)
{
	// Play a sound one time.
	OSErr error;

	if ((channel == nil) || (soundData == nil))
		return;
	
	HLock((Handle)soundData);	
	error = SndPlay(channel, soundData, kPlayAsynchronously);
	
}

void GameSound::PlayRepeatedly(void)
{
	// Loops a sound repeatedly
	
	OSErr error;

	if ((channel == nil) || (soundData == nil))
		return;
	
	HLock((Handle)soundData);	
	error = SndPlay(channel, soundData, kPlayAsynchronously);
	
	// Install the callback routine so the sound will loop
	if (error == noErr) {
		error = InstallCallback();
	}
		
}	


// Callback routines
pascal void GameSound::LoopSoundCallback(SndChannelPtr theChannel, SndCommand* theCommand)
{
	// This callback procedure tells
	// the Sound Manager to play the song
	// again, creating the looping effect.
	GameSoundPtr theSound;

	theSound = (GameSoundPtr)theCommand->param2;
	theSound->PlayRepeatedly();
}

pascal void GameSound::DisposeSoundCallback(SndChannelPtr theChannel, SndCommand* theCommand)
{
	// This callback routine tells the 
	// Sound Manager to dispose of the sound
	// when it finishes playing.
	GameSoundPtr theSound;

	theSound = (GameSoundPtr)theCommand->param2;
	theSound->Dispose();
}


OSErr GameSound::InstallCallback(void)
{
	OSErr error;
	SndCommand commandToPerform;
	
	commandToPerform.cmd = callBackCmd;
	commandToPerform.param1 =  kSoundManagerID;  
	commandToPerform.param2 = (long) this;
	
	// True means to return a result queueFull if the queue is full.
	error = SndDoCommand(channel, &commandToPerform, true);
	return error;

}

void GameSound::SetVolume(short leftVolume, short rightVolume)
{
	// Sets the sound's volume.  You pass the volume for the left
	// and right channels.  The value for the left and right channel
	// volume can range from 0 to 256.
	
	SndCommand commandToExecute;

	// Combine left and right channel values in the param2 field
	// The high 16 bits are the right channel's volume, and
	// the low 16 bits are the left channel's volume.
	commandToExecute.cmd = volumeCmd;
	commandToExecute.param1 = 0;
	commandToExecute.param2 = (rightVolume << 16) | leftVolume;

	SndDoImmediate(channel, &commandToExecute);
}

void GameSound::Pause(void)
{
	OSErr error;
	SndCommand commandToExecute;
	
	// Save the current sample rate
	commandToExecute.cmd = getRateMultiplierCmd;
	commandToExecute.param1 = 0;
	commandToExecute.param2 = (long)&sampleRate;
	error = SndDoImmediate(channel, &commandToExecute);
	
	if (error != noErr)
		return;
		
	// Now pause by setting the sample rate to 0
	commandToExecute.cmd = rateMultiplierCmd;
	commandToExecute.param1 = 0;
	commandToExecute.param2 = (long)kPauseSampleRate;	
	error = SndDoImmediate(channel, &commandToExecute);
}

void GameSound::Resume(void)
{
	OSErr error;
	SndCommand commandToExecute;
	
	commandToExecute.cmd = rateMultiplierCmd;
	commandToExecute.param1 = 0;
	commandToExecute.param2 = kNormalSampleRate;		
	error = SndDoImmediate(channel, &commandToExecute);
}

void GameSound::Stop(void)
{
	OSErr error;
	SndCommand commandToExecute;

	// First stop the sound
	commandToExecute.cmd = quietCmd;
	error = SndDoImmediate(channel, &commandToExecute);

	// Now flush the queue so no other
	// pending sound commands happen.
	commandToExecute.cmd = flushCmd;
	error = SndDoImmediate(channel, &commandToExecute);
	

}

void GameSound::Dispose(void)
{
	OSErr error;

	DisposeSoundData();
	DisposeLowLevelData();
			
	if (channel != nil) {
		error = SndDisposeChannel(channel, kDisposeNow);
		DisposePtr((Ptr)channel);
		channel = nil;
	}
	
	if (soundFileRefNum != kFileNotOpen) {
		// Close the open file if we
		// played sound from an external sound file.
		error = FSClose(soundFileRefNum);
		soundFileRefNum = kFileNotOpen;
	}

}

void GameSound::DisposeSoundData(void)
{	
	if (soundData != nil) {
		HUnlock((Handle)soundData);	
		ReleaseResource((Handle)soundData);
		soundData = nil;
	}
}


// Functions related to playing an external sound file
// using low-level techniques

void GameSound::PrepareToPlaySoundFile(Str255 filename, long bufferSize)
{
	// This function does all the setup work we need to do
	// before we can play the sound file.
	AllocateBuffers(bufferSize);
	OpenSoundFile(filename);
	FindStartOfSoundData();
	ReadSoundData(buffer1);
}

void GameSound::PlayMultiBuffered(void)
{
	QueueSoundFrame();
}

void GameSound::AllocateBuffers(long bufferSize)
{
	// When allocating buffers, make sure the buffer size
	// does not exceed the size of the sound you want to play.
	// If you make the buffer larger than the sound, static
	// will play when you reach the end of the sound.
	buffer1 = NewPtr(bufferSize);
	buffer2 = NewPtr(bufferSize);
}

void GameSound::OpenSoundFile(Str255 filename)
{
	OSErr error;
    FSSpec soundFile;

    CFBundleRef gameBundle = CFBundleGetMainBundle();
    CFURLRef resourceFileLocation;
    
    CFStringRef resourceFileToFind; 
    resourceFileToFind = CFStringCreateWithPascalString(nil, filename, 
            CFStringGetSystemEncoding());
    
    CFStringRef resourceFileType = nil;
    resourceFileLocation = CFBundleCopyResourceURL(gameBundle, resourceFileToFind,
        resourceFileType, nil); 
    
    if (resourceFileLocation == nil)
        return;
        
    FSRef fileRef;
    Boolean success = CFURLGetFSRef(resourceFileLocation, &fileRef);
    
    if (!success)
        return;
        
    // Convert the FSRef to an FSSpec so we can open the file.
    // The first nil says we don't care about catalog info.
    // The second nil says we don't care about the file's name.
    // The third nil says we don't care about the parent directory.
    error = FSGetCatalogInfo(&fileRef, kFSCatInfoGettableInfo, nil, nil,
            &soundFile, nil);
    if (error != noErr)
        return;
        
	// Now open the file
	error = FSpOpenDF(&soundFile, fsRdPerm, &soundFileRefNum);
    
    // Dispose of the Core Foundation memory we
    // allocated earlier in the function
    CFRelease(resourceFileToFind);
    CFRelease(resourceFileLocation);

}

void GameSound::FindStartOfSoundData(void)
{
	SoundComponentData soundInfo;
	unsigned long numberOfSoundFrames;
	OSErr error;

	// startOfSoundData is a data member of the GameSound class.
	error = ParseAIFFHeader(soundFileRefNum, &soundInfo, &numberOfSoundFrames, &startOfSoundData);
	
	// Set file position to the beginning of the sound data.
	SetFPos(soundFileRefNum, fsFromStart, startOfSoundData);

	// Initialize our sound header using the information
	// we got from ParseAIFFHeader().
	InitializeSoundHeader(&soundInfo, numberOfSoundFrames);
}

void GameSound::InitializeSoundHeader(SoundComponentDataPtr soundInfo, unsigned long numberOfFrames)
{
	soundHeader = new ExtSoundHeader;
	if (soundHeader == nil)
		ExitToShell();

	// Need this value to set the AIFFSampleRate properly.
	// A Float80 is an 80 bit fixed point number so using
	// dummyValue = 0.0; 
	// generates a compiler error
	Float80 dummyValue;
	dummyValue.exp = 0;
	dummyValue.man[0] = 0;
	dummyValue.man[1] = 0;
	dummyValue.man[2] = 0;
	dummyValue.man[3] = 0;
	
	// Have the header point to buffer 1 initially
	soundHeader->samplePtr = buffer1;

	soundHeader->numChannels =  soundInfo->numChannels;
	soundHeader->sampleRate = soundInfo->sampleRate;
	soundHeader->loopStart = 0;
	soundHeader->loopEnd = 0;
	soundHeader->encode = extSH;
	soundHeader->baseFrequency = kMiddleC;
	soundHeader->AIFFSampleRate = dummyValue;
	soundHeader->markerChunk = 0;
	soundHeader->instrumentChunks = 0;
	soundHeader->AESRecording = 0;
	soundHeader->sampleSize = soundInfo->sampleSize;
	soundHeader->futureUse1 = 0;
	soundHeader->futureUse2 = 0;
	soundHeader->futureUse3 = 0;
	soundHeader->futureUse4 = 0;
	soundHeader->sampleArea[0] = 0;

	// Now calculate the number of frames that our buffer can hold.

	// Find the size of the buffer.  I assume that
	// both buffers are the same size.
	long bufferSize = GetPtrSize(buffer1);
	long bytesPerSample = soundInfo->sampleSize / 8;
	long sizeOfFrame = soundInfo->numChannels * bytesPerSample;
	long framesInBuffer = bufferSize / sizeOfFrame;

	soundHeader->numFrames = framesInBuffer;
}

void GameSound::ReadSoundData(Ptr soundBuffer)
{	
	// soundBuffer will be either buffer1 or buffer2

	long currentFilePosition;
	long endOfFilePosition;
	long bytesToRead;
	OSErr error;

	error = GetFPos(soundFileRefNum, &currentFilePosition);
	if (error != noErr)
		return;

	error = GetEOF(soundFileRefNum, &endOfFilePosition);
	if (error != noErr)
		return;

	// If we reach the end of file, go back to the start of the file.
	if (currentFilePosition >= endOfFilePosition) {
		SetFPos(soundFileRefNum, fsFromStart, startOfSoundData);
	}

	bytesToRead = GetPtrSize(soundBuffer);
	long spaceLeftInFile = endOfFilePosition - currentFilePosition;
	
	// Check if theres enough room left on the file to read a whole buffer
	// of data.  If not, read whats left on the file
	if (spaceLeftInFile < bytesToRead)
		bytesToRead = spaceLeftInFile;

	// Calculate the number of frames of sound data to read.
	// Most of the time, it will use the buffer size, but if we
	// hit the end of the file, it will use the space remaining in the file.
	long bufferSize = bytesToRead;
	long bytesPerSample = soundHeader->sampleSize / 8;
	long sizeOfFrame = soundHeader->numChannels * bytesPerSample;
	long framesInBuffer = bufferSize / sizeOfFrame;

	soundHeader->numFrames = framesInBuffer;

	// Now  read the file
	error = FSRead(soundFileRefNum, &bytesToRead, soundBuffer);	
}

void GameSound::QueueSoundFrame(void)
{
	SndCommand commandToExecute;
	OSErr error;

	// Queue the buffer command
	commandToExecute.cmd = bufferCmd;
	commandToExecute.param1 = 0;
	commandToExecute.param2 = (long) soundHeader;
	error = SndDoCommand(channel, &commandToExecute, true);

	// Queue the callback command
	commandToExecute.cmd = callBackCmd;
	commandToExecute.param1 =  kSoundManagerID;  
	commandToExecute.param2 = (long) this;
	error = SndDoCommand(channel, &commandToExecute, true);
}

pascal void GameSound::MultiBufferSoundCallback(SndChannelPtr theChannel, SndCommand* theCommand)
{
	GameSoundPtr theSound;

	theSound = (GameSoundPtr)theCommand->param2;

	theSound->SwapBuffers();
	theSound->QueueSoundFrame();		
}

void GameSound::SwapBuffers(void)
{
	// Were we currently playing sound in buffer 1?
	if (soundHeader->samplePtr == buffer1) {
		// Play buffer 2 and read more data into buffer 1.
		soundHeader->samplePtr = buffer2;
		ReadSoundData(buffer1);
	}
	else {
		// Play buffer 1 and read more data into buffer 2.
		soundHeader->samplePtr = buffer1;
		ReadSoundData(buffer2);
	}
}

void GameSound::DisposeLowLevelData(void)
{	
	if (buffer1 != nil) {
		DisposePtr(buffer1);
		buffer1 = nil;
	}
	
	if (buffer2 != nil) {
		DisposePtr(buffer2);
		buffer2 = nil;
	}
	
	if (soundHeader != nil) {
		delete soundHeader;
		soundHeader = nil;
	}
}