// GamePlayer.cp		Written by Mark Szymczyk

#ifndef GAME_PLAYER
	#include "GamePlayer.h"
#endif

// Constructor
GamePlayer::GamePlayer(void)
{
    SetTotalHitPoints(kPlayerStartHP);
	SetCurrentHitPoints(kPlayerStartHP);
	SetScore(0);
	
	SetAmmo(kStartCrossbowAmmo);
	
	//SetAcceleration(kPlayerAcceleration);
	SetMaxSpeed(kPlayerMaxSpeed);
}

// Destructor
GamePlayer::~GamePlayer(void)
{

}

// Accessors
short GamePlayer::GetTotalHitPoints(void) 		
{ 
    return totalHitPoints; 
}

void GamePlayer::SetTotalHitPoints(short hp)	
{ 
    totalHitPoints = hp; 
}
		
short GamePlayer::GetCurrentHitPoints(void) 	
{ 
    return currentHitPoints; 
}

void GamePlayer::SetCurrentHitPoints(short hp) 	
{ 
    currentHitPoints = hp; 
}

long GamePlayer::GetScore(void) 
{ 
    return score; 
}
		
void GamePlayer::SetScore(long scoreValue) 
{ 
    score = scoreValue; 
}

GameWeapon GamePlayer::GetCurrentWeapon(void) 					
{ 
    return currentWeapon; 
}

void GamePlayer::SetCurrentWeapon(GameWeapon theWeapon) 	
{ 
    currentWeapon = theWeapon; 
}

Rect GamePlayer::GetViewRect(void) 			
{ 
    return viewRect; 
}
		
void GamePlayer::SetViewRect(Rect theRect)	
{ 
    viewRect = theRect; 
}

short GamePlayer::GetAmmo(void) 			
{ 
    return ammo; 
}

void GamePlayer::SetAmmo(short ammoValue)	
{ 
    ammo = ammoValue; 
}

// View Rectangle functions
void GamePlayer::DetermineViewRect(GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	Rect newViewRect;
	
	newViewRect.top = ComputeUpViewRect(theLevel, theTiles);
	newViewRect.bottom = ComputeDownViewRect(theLevel, theTiles);
	newViewRect.left = ComputeLeftViewRect(theLevel, theTiles);
	newViewRect.right = ComputeRightViewRect(theLevel, theTiles);
	
	SetViewRect(newViewRect);
	
}

short GamePlayer::ComputeUpViewRect(GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	short closestWall;
	short horizontalCenter = (worldX + (spriteWidth / 2))/ kTileWidth;
	short verticalCenter = (worldY + (spriteHeight / 2))/ kTileHeight;

	short startX = horizontalCenter;
	short startY = verticalCenter; //worldY / kTileHeight;
	
	closestWall = FindNearestSolidUp(startX, startY, theLevel, theTiles, kSkipOneTile);	
	// We assume walls and doors are 3 tiles thick.
	// We subtract 2 so the player can see the walls.
	short result = (closestWall - 2) * kTileHeight;
	return result;
}

short GamePlayer::TraverseForSolidUp(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short threshold)
{
	// startX and startY is where we begin our traversal
	// The threshold keeps us from doing unnecessary traversals.
	// This function will be called by FindNearestWallUp and the threshold
	// will keep us from going past the previously computed nearest wall.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	short rowToTest = startY;
	short columnToTest = startX;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = theLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTiles->tileTable[tileNum].GetTileType();

	// We move up one tile at a
	// time until we hit a wall or door tile,
	// or we reach the threshold.
	while ((tileAttribute != kWallTile) && (tileAttribute != kDoorTile) && 
            (rowToTest > threshold)) {
		rowToTest--;
		mapIndex = (rowToTest * theLevelWidth) + columnToTest;
		tileNum = theLevel->levelMap[mapIndex].GetValue();	
		tileAttribute = theTiles->tileTable[tileNum].GetTileType();
	}
		
	return (rowToTest);

}

short GamePlayer::FindNearestSolidUp(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short skipValue)
{
	short leftEdge = 0;
	short rightEdge = theLevel->GetLevelWidth();
	
	// Determine the range we have to traverse
	short startColumn = TraverseForSolidLeft(startX, startY, theLevel, 
            theTiles, leftEdge);
	// Start at first floor tile.
	// TraverseForSolid returns the first wall.
	startColumn++;
	
	short endColumn = TraverseForSolidRight(startX, startY, theLevel, 
            theTiles, rightEdge);
	// End at last floor tile
	// TraverseForSolid will return the ending wall.
	// The second decrement is to properly handle doorways
	endColumn--;
	endColumn--;
	
	// Bounds checking
	if (endColumn < startColumn)
		return startY;
			
	short nearestWall = 0;
	short currentWall = 0;
	
	// Traverse up along the entire range looking for the nearest wall.
	for (short column = startColumn; column <= endColumn; column += skipValue) {
		currentWall = TraverseForSolidUp(column, startY, theLevel, theTiles, nearestWall);
		if (currentWall > nearestWall) {
			nearestWall = currentWall;
		}
	}
	
	return (nearestWall);
}
		
			
short GamePlayer::ComputeDownViewRect(GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	short closestWall;
	short horizontalCenter = (worldX + (spriteWidth / 2))/ kTileWidth;
	short verticalCenter = (worldY + (spriteHeight / 2))/ kTileHeight;
	short startX = horizontalCenter;
	short startY = verticalCenter; //(worldY + spriteHeight) / kTileHeight;
	
	closestWall = FindNearestSolidDown(startX, startY, theLevel, theTiles, kSkipOneTile);
	
	// We assume walls and doors are 3 tiles thick.
	// We add 2 so the player can see the walls.
	short result = (closestWall + 2) * kTileHeight;
	return result;

}

short GamePlayer::TraverseForSolidDown(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short threshold)
{
	// startX and startY is where we begin our traversal
	// The threshold keeps us from doing unnecessary traversals.
	// This function will be called by FindNearestWallDown and the threshold
	// will keep us from going past the previously computed nearest wall.

	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	short rowToTest = startY;
	short columnToTest = startX;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = theLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTiles->tileTable[tileNum].GetTileType();

	// We move down one tile at a
	// time until we hit a wall or door tile,
	// or we hit the threshold.
	while ((tileAttribute != kWallTile) && (tileAttribute != kDoorTile) && 
            (rowToTest < threshold)) {
		rowToTest++;
		mapIndex = (rowToTest * theLevelWidth) + columnToTest;
		tileNum = theLevel->levelMap[mapIndex].GetValue();	
		tileAttribute = theTiles->tileTable[tileNum].GetTileType();
	}
		
	return (rowToTest);

}

short GamePlayer::FindNearestSolidDown(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short skipValue)
{
	short leftEdge = 0;
	short rightEdge = theLevel->GetLevelWidth();
	
	// Determine the range we have to traverse
	short startColumn = TraverseForSolidLeft(startX, startY, theLevel, 
            theTiles, leftEdge);
	// Start at first floor tile.
	// TraverseForSolid returns the first wall.
	startColumn++;
	
	short endColumn = TraverseForSolidRight(startX, startY, theLevel, 
            theTiles, rightEdge);
	// End at last floor tile
	// TraverseForSolid will return the ending wall.
	// The second decrement is to properly handle doorways
	endColumn--;
	endColumn--;

	// Bounds checking
	if (endColumn < startColumn)
		return startY;
	
	short nearestWall = theLevel->GetLevelHeight();
	short currentWall = 0;
	
	// Traverse down the entire range looking for the nearest wall.
	for (short column = startColumn; column <= endColumn; column += skipValue) {
		currentWall = TraverseForSolidDown(column, startY, theLevel, 
                theTiles, nearestWall);
		if (currentWall < nearestWall) {
			nearestWall = currentWall;
		}
	}
	
	return (nearestWall);
}

	
short GamePlayer::ComputeLeftViewRect(GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	short closestWall;
	short horizontalCenter = (worldX + (spriteWidth / 2))/ kTileWidth;
	short verticalCenter = (worldY + (spriteHeight / 2))/ kTileHeight;
	short startX = horizontalCenter; //worldX / kTileWidth;
	short startY = verticalCenter;
	closestWall = FindNearestSolidLeft(startX, startY, theLevel, theTiles, kSkipOneTile);
	
	// We assume walls and doors are 3 tiles thick.
	// We subtract 2 so the player can see the walls.
	short result = (closestWall - 2) * kTileWidth;
	return result;
}

short GamePlayer::TraverseForSolidLeft(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short threshold)
{
	// startX and startY is where we begin our traversal
	// The threshold keeps us from doing unnecessary traversals.
	// This function will be called by FindNearestWallLeft and the threshold
	// will keep us from going past the previously computed nearest wall.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	short rowToTest = startY;
	short columnToTest = startX;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = theLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTiles->tileTable[tileNum].GetTileType();

	// We move left one tile at a
	// time until we hit a wall or door tile,
	// or we hit the threshold.
	while ((tileAttribute != kWallTile) && (tileAttribute != kDoorTile) && 
            (columnToTest > threshold)) {
		columnToTest--;
		mapIndex = (rowToTest * theLevelWidth) + columnToTest;
		tileNum = theLevel->levelMap[mapIndex].GetValue();	
		tileAttribute = theTiles->tileTable[tileNum].GetTileType();
	}
		
	return (columnToTest);

}

short GamePlayer::FindNearestSolidLeft(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short skipValue)
{
	short topEdge = 0;
	short bottomEdge = theLevel->GetLevelHeight();
	
	// Determine the range we have to traverse
	short startRow = TraverseForSolidUp(startX, startY, theLevel, theTiles, topEdge);
	// Start at first floor tile.
	// TraverseForSolid returns the first wall.
	startRow++;
	
	short endRow = TraverseForSolidDown(startX, startY, theLevel, theTiles, bottomEdge);
	// End at last floor tile
	// TraverseForSolid will return the ending wall.
	// The second decrement is to properly handle doorways
	endRow--;
	endRow--;

	// Bounds checking
	if (endRow < startRow)
		return startX;
	
	short nearestWall = 0;
	short currentWall = 0;
	
	// Traverse left across the entire range looking for the nearest wall.
	for (short row = startRow; row <= endRow; row += skipValue) {
		currentWall = TraverseForSolidLeft(startX, row, theLevel, theTiles, nearestWall);
		if (currentWall > nearestWall) {
			nearestWall = currentWall;
		}
	}
	
	return (nearestWall);
}


short GamePlayer::ComputeRightViewRect(GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	short closestWall;
	short horizontalCenter = (worldX + (spriteWidth / 2))/ kTileWidth;
	short verticalCenter = (worldY + (spriteHeight / 2)) / kTileHeight;
	short startX = horizontalCenter; //(worldX + spriteWidth) / kTileWidth;
	short startY = verticalCenter;
	
	closestWall = FindNearestSolidRight(startX, startY, theLevel, theTiles, kSkipOneTile);
	
	// We assume walls and doors are 3 tiles thick.
	// We add 2 so the player can see the walls.
	short result = (closestWall + 2) * kTileWidth;
	return result;}
		
short GamePlayer::TraverseForSolidRight(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short threshold)
{
	// startX and startY is where we begin our traversal
	// The threshold keeps us from doing unnecessary traversals.
	// This function will be called by FindNearestWallLeft and the threshold
	// will keep us from going past the previously computed nearest wall.
	
	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	short rowToTest = startY;
	short columnToTest = startX;
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = theLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTiles->tileTable[tileNum].GetTileType();
	
	// We move right one tile at a
	// time until we hit a wall or door tile,
	// or we hit the threshold.
	while ((tileAttribute != kWallTile) && (tileAttribute != kDoorTile) && 
            (columnToTest < threshold)) {
		columnToTest++;
		mapIndex = (rowToTest * theLevelWidth) + columnToTest;
		tileNum = theLevel->levelMap[mapIndex].GetValue();	
		tileAttribute = theTiles->tileTable[tileNum].GetTileType();
	}
		
	return (columnToTest);

}

short GamePlayer::FindNearestSolidRight(short startX, short startY, GameLevelPtr theLevel,
        GameTileListPtr theTiles, short skipValue)
{
	short topEdge = 0;
	short bottomEdge = theLevel->GetLevelHeight();
	
	// Determine the range we have to traverse
	short startRow = TraverseForSolidUp(startX, startY, theLevel, theTiles, topEdge);
	// Start at first floor tile.
	// TraverseForSolid returns the first wall.
	startRow++;
	
	short endRow = TraverseForSolidDown(startX, startY, theLevel, theTiles, bottomEdge);
	// End at last floor tile
	// TraverseForSolid will return the ending wall.
	// The second decrement is to properly handle doorways
	endRow--;
	endRow--;

	// Bounds checking
	if (endRow < startRow)
		return startX;
	
	short nearestWall = theLevel->GetLevelWidth();
	short currentWall = 0;
	
	// Traverse left across the entire range looking for the nearest wall.
	for (short row = startRow; row <= endRow; row += skipValue) {
		currentWall = TraverseForSolidRight(startX, row, theLevel, theTiles, nearestWall);
		if (currentWall < nearestWall) {
			nearestWall = currentWall;
		}
	}
	
	return (nearestWall);
}



// Event related functions
void GamePlayer::HandleChangeWeapon(void)
{

}

// Taking of items
Boolean GamePlayer::TakeItem(GameItemPtr theItem)
{
	switch (theItem->GetWhatItIs()) {
		case kBolts:
			return (TakeBolts());
			break;

		case kCoins:
			return (TakeCoins());
			break;

		case kGems:
			return (TakeGems());
			break;

		case kHealingPotion:
			return (TakeHealingPotion());
			break;

		default:
		 	return false;
		 	break;
	}
	
	return false;
}

Boolean GamePlayer::TakeBolts(void)
{
	if (ammo >= kMaximumCrossbowAmmo) {
		return false;
	}
	else {
		ammo = ammo + kIncrementCrossbowAmmo;
		if (ammo > kMaximumCrossbowAmmo)
			ammo = kMaximumCrossbowAmmo;
		return true;
	}
}

Boolean GamePlayer::TakeCoins(void)
{
	score = score + kCoinsScore;
	return true;
}

Boolean GamePlayer::TakeGems(void)
{
	score = score + kGemsScore;
	return true;
}

Boolean GamePlayer::TakeHealingPotion(void)
{
	if (currentHitPoints == totalHitPoints) {
		return false;
	}
	else {
		currentHitPoints = currentHitPoints + kHealingPotionHealingValue;
		if (currentHitPoints > totalHitPoints)
			currentHitPoints = totalHitPoints;
		return true;
	}
	
}


// Damage related functions
void GamePlayer::TakeHit(GameWeaponPtr weaponThatHit)
{
	currentHitPoints = currentHitPoints - weaponThatHit->GetDamage();
}

Boolean GamePlayer::WasKilled(void)
{
	if (currentHitPoints <= 0)
		return true;
	else
		return false;
}			


// Miscellaneous and utility functions
Boolean GamePlayer::CanAttack(void)
{
    if (ammo > 0)
        return true;
    else
        return false;
        
}

Boolean GamePlayer::LowOnAmmo(void)
{
	if (ammo < kLowAmmoThreshold)
        return true;
    else
        return false;        
}

void GamePlayer::DetermineCurrentWeapon(void)
{
    // The player can use only a crossbow so
    // there's not much to determine.
    currentWeapon.SetWeaponClass(kEdgedWeapon);
    currentWeapon.SetRange(kCrossbowRange);
    currentWeapon.SetDamage(kCrossbowDamage);
}

void GamePlayer::FireMissile(void)
{
    ammo--;
}


Boolean GamePlayer::ReachedExit(GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	// Check if the center of the player sprite
	// is on an exit tile

	short tileNum;
	GameTileType tileAttribute;
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	short playerCenterX = worldX + (GetSpriteWidth() / 2);
	short playerCenterY = worldY + (GetSpriteHeight() / 2);
	
	short rowToTest = playerCenterY / kTileHeight;
	short columnToTest = playerCenterX / kTileWidth;
	
	mapIndex = (rowToTest * theLevelWidth) + columnToTest;
	tileNum = theLevel->levelMap[mapIndex].GetValue();	
	tileAttribute = theTiles->tileTable[tileNum].GetTileType();	
	if (tileAttribute == kExitTile) 
		return true;
	else
		return false;

}

short GamePlayer::GetFramesInSequence(void)
{
	int playerAction = GetAction();
	
	switch (playerAction) {
		case kSpriteMoving:
			return kPlayerMoveFrames;
			break;
			
		case kSpriteFighting:
			return kPlayerFightFrames;
			break;
			
		default:
			return 0;
	}
}

void GamePlayer::DetermineSpriteToDraw(void)
{
	// For the player, this function will determine what column
	// the sprite to draw should be.  The row is determined by
	// the direction the player is facing.  The column
	// will be determined by what action the player is performing and
	// what frame we're at in a multi-frame action.
	
	short currentRow;
	
	switch (direction) {
		case kSpriteFacingUp:
			currentRow = kPlayerUpFrame;
			break;
			
		case kSpriteFacingDown:
			currentRow = kPlayerDownFrame;
			break;
			
		case kSpriteFacingLeft:
			currentRow = kPlayerLeftFrame;
			break;
			
		case kSpriteFacingRight:
			currentRow = kPlayerRightFrame;
			break;
			
		default:
			break;
		
	}

	SetSpriteRow(currentRow);

	// Now calculate sprite column
	short currentColumn;
	
	switch (action) {
		case kSpriteStanding:
			currentColumn = kPlayerStandFrame;
			break;
			
		case kSpriteMoving:	
			currentColumn = kPlayerMoveFrame + frame;
			break;
			
		case kSpriteFighting:
			currentColumn = kPlayerAttackFrame + frame;
			break;
			
		default:
			break;
			
	}

	SetSpriteColumn(currentColumn);

    /*
    // Now determine the column
    short currentColumn;
    
	switch (action) {
		case kSpriteMoving:	
			break;
			
		case kSpriteFighting:
			currentColumn = currentColumn + kPlayerAttackFrame;
			break;
			
		default:
			break;
			
	}
	
	currentColumn = currentColumn + frame;
	SetSpriteColumn(currentColumn);
	*/
}	
