// GameOffscreenBuffer.cp		Written by Mark Szymczyk
// Book Version

#ifndef GAME_OFFSCREEN_BUFFER
	#include "GameOffscreenBuffer.h"
#endif

// Constructor
GameOffscreenBuffer::GameOffscreenBuffer(void) 
{
	bufferStorage = nil;
}


// Accessors
GWorldPtr GameOffscreenBuffer::GetBufferStorage(void)
{ 
	return bufferStorage; 
}

void GameOffscreenBuffer::SetBufferStorage (GWorldPtr theStorage)
{ 
	bufferStorage = theStorage; 
}


// Member functions
Boolean GameOffscreenBuffer::Create(short width, short height, short colorDepth,
									CTabHandle colorTable, GWorldFlags flags)
{

	// Create an offscreen buffer with the supplied
	// width, height, color depth, color table, and flags. 
	// If you are using a color depth of 16 or 32, pass nil
	// for the color table. 
	// Returns true if the buffer is created successfully.
	
	QDErr error;
	
	Rect bufferRect;
	bufferRect.top = 0;
	bufferRect.bottom = height;
	bufferRect.left = 0;
	bufferRect.right = width;
	
	// The nil means there's no GDevice, which there shouldn't
	// be because the buffer is offscreen.
	error = NewGWorld(&bufferStorage, colorDepth, &bufferRect, colorTable, nil, flags);
	
	if ((error == noErr) && (bufferStorage != nil))
		// Buffer created successfully
		return true;
	else
		return false;
	
}

Boolean GameOffscreenBuffer::CreateInVRAM(short width, short height, short colorDepth,
										CTabHandle colorTable, GWorldFlags flags)
{
	// Create an offscreen buffer in VRAM with the supplied
	// width, height, color depth, color table, and flags.  
	// Returns true if the buffer is created successfully.  
	// Assumes the user has Mac OS 9 since VRAM based GWorlds run 
	// only on Mac OS 9 and later.
	
	QDErr error;
	
	Rect bufferRect;
	bufferRect.top = 0;
	bufferRect.bottom = height;
	bufferRect.left = 0;
	bufferRect.right = width;

	// Add the create buffer in VRAM flag to whatever flags you
	// supply in the flags parameter.
	flags |= useDistantHdwrMem;
	
	// The nil means there's no GDevice, which there shouldn't
	// be because the buffer is offscreen.  
	error = NewGWorld(&bufferStorage, colorDepth, &bufferRect, colorTable, nil, flags);
	
	if ((error == noErr) && (bufferStorage != nil))
		// Buffer created successfully
		return true;
	else
		return false;
	
}

Boolean GameOffscreenBuffer::CreatePICTSized(short pictureResourceID, short colorDepth,
									CTabHandle colorTable, GWorldFlags flags)
{

	// Create an offscreen buffer the size of a PICT. 
	// If you are using a color depth of 16 or 32, pass nil
	// for the color table. 
	// Returns true if the buffer is created successfully.
	
	QDErr error;
	
	Rect bufferRect;

	PicHandle thePicture = GetPicture(pictureResourceID);
	if(thePicture == nil)
		return false;
		
	bufferRect = (**thePicture).picFrame;
	
	// The nil means there's no GDevice, which there shouldn't
	// be because the buffer is offscreen.
	error = NewGWorld(&bufferStorage, colorDepth, &bufferRect, colorTable, nil, flags);
	
	if ((error == noErr) && (bufferStorage != nil))
		// Buffer created successfully
		return true;
	else
		return false;
	
}

Boolean GameOffscreenBuffer::Update(short width, short height, short colorDepth,
									CTabHandle colorTable, GWorldFlags flags)
{

	// Update an existing offscreen buffer with the supplied
	// width, height, color depth, color table, and flags. 
	// If you need to resize an offscreen buffer or change the 
	// color table, updating the buffer is faster than creating
	// a new one.
	 
	// Returns true if the buffer is updated successfully.
	
	GWorldFlags error;
	
	Rect bufferRect;
    bufferRect.top = 0;
	bufferRect.bottom = height;
	bufferRect.left = 0;
	bufferRect.right = width;
	
	// Check if the buffer exists.  If not, we'll create a buffer.
	if (bufferStorage == nil)
		Create(width, height, colorDepth, colorTable, flags);
			
	// The nil means there's no GDevice, which there shouldn't
	// be because the buffer is offscreen.
	error = UpdateGWorld(&bufferStorage, colorDepth, &bufferRect, colorTable, nil, flags);
	
	if ((error == noErr) && (bufferStorage != nil))
		// Buffer created successfully
		return true;
	else
		return false;
	
}

void GameOffscreenBuffer::Dispose(void)
{
	if (bufferStorage != nil) {
		DisposeGWorld(bufferStorage);
		bufferStorage = nil;
	}
}

void GameOffscreenBuffer::Draw(short pictureResourceID)
{
	// Draws the picture into storage.
	
	//OSStatus error;
	PicHandle picToOpen = GetPicture(pictureResourceID);
	
	CGrafPtr oldPort;
	GDHandle oldGDevice;
	Boolean canLockPixels;
			
	// Save previous drawing area.
	GetGWorld (&oldPort, &oldGDevice);

	// Have drawing done to bufferStorage.
	SetGWorld(bufferStorage, nil);
			
	// Lock the buffer storage
	PixMapHandle thePixMap = GetGWorldPixMap(bufferStorage);
	if (thePixMap != nil)
		canLockPixels = LockPixels(thePixMap);
	else
		canLockPixels = false;

	// Draw the image in the offscreen buffer.
    OSErr error;
	PictInfo pictureData;
    Rect destRect;
    if (canLockPixels){	
        // Get the size of the picture
        
		// In Carbon, we can't access the pic's frame directly.
        // We must call GetPictInfo(), then access the sourceRect
        // member of the PictInfo data structure.  The returnColorTable
        // value tells us to return a color table instead of a palette.
        // the 256 tells us to return 256 colors in the table.
        // The systemMethod value tells QuickDraw to handle the colors.
        // The last parameter should be 0.
        error = GetPictInfo(picToOpen, &pictureData, returnColorTable, 
                256, systemMethod, 0);
        
        destRect = pictureData.sourceRect;
		DrawPicture(picToOpen, &destRect);
		UnlockPixels(thePixMap);

        // Dispose of the memory GetPictInfo() allocates
        DisposeCTable(pictureData.theColorTable);
	}
		
	// restore graphics port and GDevice
	SetGWorld(oldPort, oldGDevice);

	// Clean Up
	//ReleaseResource(Handle(picToOpen));
    KillPicture(picToOpen);
}

short GameOffscreenBuffer::GetPixelValue(short row, short column)
{
	// Returns the indexed color value for the pixel at
	// the given row and column.  Can be used for 8 bit
	// color or lower.
	
	PixMapHandle bufferPixelMap = GetGWorldPixMap(bufferStorage);
	Ptr bufferBaseAddress =  GetPixBaseAddr(bufferPixelMap);
	short bufferRowBytes = (**bufferPixelMap).rowBytes & kRowByteMask;
	
	long bufferIndex = (bufferRowBytes * row) + column;
	return bufferBaseAddress[bufferIndex];
	
}
