// GameContext.cp 		Written by Mark Szymczyk

// Make sure GameContext.h is included just once
#ifndef GAME_CONTEXT
	#include "GameContext.h"
#endif

GameContext::GameContext(void)
{
	hOffset = 0;
	vOffset = 0;
	background = new GameOffscreenBuffer;
	tileStorage = new GameOffscreenBuffer;
    playerSpriteStorage = new GameOffscreenBuffer;
    enemySpriteStorage = new GameOffscreenBuffer;
    weaponSpriteStorage = new GameOffscreenBuffer;
    itemSpriteStorage = new GameOffscreenBuffer;
    dirtyRectTable = nil;
}

GameContext::~GameContext(void)
{
	CleanUp();
}

// Accessors
GameOffscreenBufferPtr GameContext::GetBackground(void)
{
	return background;
}
		
void GameContext::SetBackground(GameOffscreenBufferPtr theBackground)
{
	background = theBackground;
}

GameOffscreenBufferPtr GameContext::GetTileStorage(void)
{
	return tileStorage;
}
		
void GameContext::SetTileStorage (GameOffscreenBufferPtr theStorage)
{
	tileStorage = theStorage; 
}

GameOffscreenBufferPtr GameContext::GetPlayerSpriteStorage(void)
{
	return playerSpriteStorage;
}
		
void GameContext::SetPlayerSpriteStorage (GameOffscreenBufferPtr theStorage)
{
	playerSpriteStorage = theStorage; 
}

GameOffscreenBufferPtr GameContext::GetEnemySpriteStorage(void)
{
	return enemySpriteStorage;
}
		
void GameContext::SetEnemySpriteStorage (GameOffscreenBufferPtr theStorage)
{
	enemySpriteStorage = theStorage; 
}

GameOffscreenBufferPtr GameContext::GetWeaponSpriteStorage(void)
{
	return weaponSpriteStorage;
}
		
void GameContext::SetWeaponSpriteStorage (GameOffscreenBufferPtr theStorage)
{
	weaponSpriteStorage = theStorage; 
}

GameOffscreenBufferPtr GameContext::GetItemSpriteStorage(void)
{
	return itemSpriteStorage;
}
		
void GameContext::SetItemSpriteStorage (GameOffscreenBufferPtr theStorage)
{
	itemSpriteStorage = theStorage; 
}
		
short GameContext::GetScreenWidth(void)
{
	return screenWidth;
}
		
void GameContext::SetScreenWidth(short theWidth)
{
	screenWidth = theWidth;
}

short GameContext::GetScreenHeight(void)
{
	return screenHeight;
} 
		
void GameContext::SetScreenHeight(short theHeight)
{
	screenHeight = theHeight;
}

short GameContext::GetHOffset(void)
{
	return hOffset;
}
		
void GameContext::SetHOffset(short theOffset)
{
	hOffset = theOffset;
}

short GameContext::GetVOffset(void)
{
	return vOffset;
}

void GameContext::SetVOffset(short theOffset)
{
	vOffset = theOffset;
}
		
Blitter GameContext::GetTileBlitter(void)
{
	return tileBlitter;
}
		
void GameContext::SetTileBlitter(Blitter theBlitter)
{
	tileBlitter = theBlitter;
}


void GameContext::CleanUp(void)
{
	// Have to do these nil checks so we don't accidentally
	// dispose of memory we haven't allocated yet.  
	
	if (background != nil)
		DisposeBackground();
		
	if (tileStorage != nil)
		DisposeTileStorage();
	
	DisposeSpriteBuffers();
	DisposeDirtyRectTable();
}

// Drawing and Buffer related functions
Boolean GameContext::CreateBackground(short width, short height, short colorDepth,
									CTabHandle colorTable, GWorldFlags flags)
{
	Boolean result;
	
	result = background->Create(width, height, colorDepth, colorTable, flags);
	
	return result;
}

Boolean GameContext::CreateTileStorage(short width, short height, short colorDepth,
									CTabHandle colorTable, GWorldFlags flags)
{
	Boolean result;
	
	result = tileStorage->Create(width, height, colorDepth, colorTable, flags);
	
	return result;
}

Boolean GameContext::CreateSpriteBuffers(short colorDepth,
                        CTabHandle colorTable, GWorldFlags flags)
{
    Boolean result;
    
    // Create player sprite storage
    result = playerSpriteStorage->CreatePICTSized(kPlayerSpriteID, colorDepth,
                    colorTable, flags);
    
    // Enemy sprite storage
    result = enemySpriteStorage->CreatePICTSized(kEnemySpriteID, colorDepth,
                    colorTable, flags);
    
    // Weapon sprite storage
    result = weaponSpriteStorage->CreatePICTSized(kWeaponSpriteID, colorDepth,
                    colorTable, flags);
    
    // Item sprite storage
    result = itemSpriteStorage->CreatePICTSized(kItemSpriteID, colorDepth,
                    colorTable, flags);
    
    return result;
}


void GameContext::DisposeBackground(void)
{
	background->Dispose();
	
	// Just calling Dispose() leaves a 4 byte
	// memory leak so we must explicitly delete
	// the background.
	delete background;
}

void GameContext::DisposeTileStorage(void)
{
	tileStorage->Dispose();
	
	// Just calling Dispose() leaves a 4 byte
	// memory leak so we must explicitly delete
	// the tile storage.
	delete tileStorage;
}

void GameContext::DisposeSpriteBuffers(void)
{
	playerSpriteStorage->Dispose();
	enemySpriteStorage->Dispose();
	weaponSpriteStorage->Dispose();
	itemSpriteStorage->Dispose();
	
	// Just calling Dispose() leaves a 4 byte
	// memory leak so we must explicitly delete
	// the sprite storage.
	delete playerSpriteStorage;
	delete enemySpriteStorage;
	delete weaponSpriteStorage;
	delete itemSpriteStorage;
}



void GameContext::DrawBackground(GameLevelPtr theLevel)
{
	CGrafPtr savedPort;
	GDHandle savedGDevice;
	//CGrafPtr backgroundPort;
	//GDHandle backgroundGDevice;
	
	//OSStatus error;
	
	// save port and GDevice
	GetGWorld(&savedPort, &savedGDevice);
	
	// prepare to draw the background
	SetGWorld(background->GetBufferStorage(), nil);
	
	// Draw the background.  We're just drawing the upper left
	// portion of the level map.  This function will
	// just be used at the start of the level to build the background.
	short rowsInContext = screenHeight / kTileHeight;
	short columnsInContext = screenWidth / kTileWidth;
	UInt32 mapIndex;
	short theLevelWidth = theLevel->GetLevelWidth();

	for (short row = vOffset; row < (vOffset + rowsInContext); row++) {
		for (short column = hOffset; column < (hOffset + columnsInContext); column++) {
			mapIndex = (row * theLevelWidth) + column;
			if (theLevel->levelMap[mapIndex].GetValue() != kEmpty)
				DrawTileFromLevelMap(theLevel, row, column);
		}
	}
	
	
	// restore saved port and GDevice
	SetGWorld(savedPort, savedGDevice);
}

void GameContext::DrawTileStorage(short pictureID)
{
	// What we do here is draw the PICT containing the tiles
	// into tile storage.
	
	tileStorage->Draw(pictureID);
    
	// Initialize the tile blitter here
	InitializeTileBlitter();
}

void GameContext::DrawSpriteBuffers(void)
{
    playerSpriteStorage->Draw(kPlayerSpriteID);
    enemySpriteStorage->Draw(kEnemySpriteID);
    weaponSpriteStorage->Draw(kWeaponSpriteID);
    itemSpriteStorage->Draw(kItemSpriteID);
}

void GameContext::UpdateBackground(RectPtr updateRect, GameLevelPtr theLevel)
{
	// We would only use this function to respond to 
	// update events.
	Boolean backgroundChanged;
	short localRow;
	short localColumn;
	short rowsInContext = screenHeight / kTileHeight;
	short columnsInContext = screenWidth / kTileWidth;
	
	short topTile = updateRect->top / kTileHeight;
	short bottomTile = updateRect->bottom / kTileHeight;
	short leftTile = updateRect->left / kTileWidth;
	short rightTile = updateRect->right / kTileWidth;

	UInt32 mapIndex;
	short theLevelWidth = theLevel->GetLevelWidth();
	
	for (short row = topTile; row < bottomTile; row++) {
		for (short column = leftTile; column < rightTile; column++) {
			mapIndex = (row * theLevelWidth) + column;
			if (theLevel->levelMap[mapIndex].IsVisibleToPlayer() == false) {
				theLevel->levelMap[mapIndex].SetVisibleToPlayer(true);
				localRow = row - vOffset;
				localColumn = column - hOffset;
				if ((localRow < rowsInContext) && (localColumn < columnsInContext)) {
					// Have to use localRow and localColumn because
					// DrawTileFromLevelMap adds the offsets.
					DrawTileFromLevelMap(theLevel, localRow, localColumn);
					backgroundChanged = true;
				}
			}
		}
	}


		
}

	
void GameContext::DrawTileFromLevelMap(GameLevelPtr theLevel, short theRow, short theColumn)
{
	short worldRow = theRow + vOffset;
	short worldColumn = theColumn + hOffset;	
	
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex = (worldRow * theLevelWidth) + worldColumn;	
	short tileNumber = theLevel->levelMap[mapIndex].GetValue();
	
	// We want to see the values so testing the visibility is commented out
	/*
	if (theLevel->levelMap[mapIndex].IsVisibleToPlayer() == false)
		tileNumber = kBlankTileValue;
	*/
		 
	Point tileGridLocation = GetGridLocation(tileNumber);
	Rect tileRect = GetRectFromGridLocation(tileGridLocation);

	Point backgroundLocation;
	backgroundLocation.h = theColumn;
	backgroundLocation.v = theRow;
	Rect backgroundRect = GetRectFromGridLocation(backgroundLocation);

	//tileBlitter.SetSourceRect(tileRect);
	//tileBlitter.SetDestinationRect(backgroundRect);
	tileBlitter.Setup(background->GetBufferStorage(), tileRect, backgroundRect);
	tileBlitter.DrawImageToOffscreenBuffer();
	
}

void GameContext::InitializeTileBlitter(void)
{
	// Sets up the part of the tile blitter that will remain unchanged
	// from frame to frame.  For tile drawing, we want
	// to use CopyBits() srcCopy transfer mode.
	
	tileBlitter.Initialize(tileStorage->GetBufferStorage(), srcCopy);
}


// Scrolling functions
short GameContext::ScrollLeft(GameLevelPtr theLevel, short tilesToScroll)
{
    // The scrolling functions return the number 
    // of tiles scrolled
    
	// Bounds checking
	if (hOffset <= kLeftEdge)
		return 0;

	// If a full scroll puts us past the left edge
	// then only scroll to the edge.
	if ((hOffset - tilesToScroll) < kLeftEdge) {
		short amountToScroll = hOffset;
        return (ScrollLeft(theLevel, amountToScroll));
		//return amountToScroll;
	}	
		
	CGrafPtr oldPort;
	GDHandle oldGDevice;

	//OSStatus error;

	// save previous drawing area
	GetGWorld (&oldPort, &oldGDevice);

	// Set drawing area to the background
	SetGWorld (background->GetBufferStorage(), nil);
	
	// Draw the portion of the background that remains after the scroll.
	// For a scroll left, we move the leftmost columns that remain to
	// the right edge of the screen.
	Rect sourceRect;
	Rect destRect;
	
	sourceRect.top = 0;
	sourceRect.left = 0;
	sourceRect.bottom = screenHeight;
	sourceRect.right = screenWidth - (tilesToScroll * kTileWidth);
	
	destRect.top = 0;
	destRect.left = tilesToScroll * kTileWidth;
	destRect.bottom = screenHeight;
	destRect.right = screenWidth;
	
	Blitter theBlitter;
	theBlitter.SetSourceBuffer(background->GetBufferStorage());
	theBlitter.SetDestinationBuffer(background->GetBufferStorage());
	theBlitter.SetSourceRect(sourceRect);
	theBlitter.SetDestinationRect(destRect);
	theBlitter.SetDrawingMode(srcCopy);
	theBlitter.DrawImageToOffscreenBuffer();


	// Update the origin to reflect the scroll
	hOffset = hOffset - tilesToScroll;
			
	// Fill in the new tiles.  For a scroll left, it will
	// be the leftmost columns.
	short rowsInContext = screenHeight / kTileHeight;
	for (short column = 0; column < tilesToScroll; column++) {
		for (short row = 0; row < rowsInContext; row++) {
			DrawTileFromLevelMap(theLevel, row, column);
		}
	}
	
		
	// restore graphics port and GDevice
	SetGWorld(oldPort, oldGDevice);
		
    return tilesToScroll;
}

short GameContext::ScrollRight(GameLevelPtr theLevel, short tilesToScroll)
{
    // The scrolling functions return the number 
    // of tiles scrolled

	// Bounds checking.  If we've hit the right edge of the level, don't
	// bother scrolling.
	short rightEdge = theLevel->GetLevelWidth() - (screenWidth / kTileWidth);
	if (hOffset >= rightEdge)
		return 0;

	// If a full scroll puts us past the right edge
	// then only scroll to the edge.
	if ((hOffset + tilesToScroll) > rightEdge) {
		short amountToScroll = rightEdge - hOffset;
		return (ScrollRight(theLevel, amountToScroll));
		//return amountToScroll;	
	}

	CGrafPtr oldPort;
	GDHandle oldGDevice;

	// save previous drawing area
	GetGWorld (&oldPort, &oldGDevice);

	// Set drawing area to the background
	SetGWorld (background->GetBufferStorage(), nil);
	
	// Draw the portion of the background that remains after the scroll
	// For a scroll right, we move the rightmost columns that remain to
	// the left edge of the screen.
	Rect sourceRect;
	Rect destRect;
	
	sourceRect.top = 0;
	sourceRect.left = tilesToScroll * kTileWidth;
	sourceRect.bottom = screenHeight;
	sourceRect.right = screenWidth;
	
	destRect.top = 0;
	destRect.left = 0;
	destRect.bottom = screenHeight;
	destRect.right = screenWidth - (tilesToScroll * kTileWidth);

	Blitter theBlitter;
	theBlitter.SetSourceBuffer(background->GetBufferStorage());
	theBlitter.SetDestinationBuffer(background->GetBufferStorage());
	theBlitter.SetSourceRect(sourceRect);
	theBlitter.SetDestinationRect(destRect);
	theBlitter.SetDrawingMode(srcCopy);
	theBlitter.DrawImageToOffscreenBuffer();


	// Update the origin to reflect the scroll
	hOffset = hOffset + tilesToScroll;
			
	// Fill in the new tiles.  For a scroll right, it will be
	// the rightmost columns
	short rowsInContext = screenHeight / kTileHeight;
	short columnsInContext = screenWidth / kTileWidth;
	
	for (short column = columnsInContext - tilesToScroll; column < columnsInContext; column++) {
		for (short row = 0; row < rowsInContext; row++) {
			DrawTileFromLevelMap(theLevel, row, column);
		}
	}
	
					
	// restore graphics port and GDevice
	SetGWorld(oldPort, oldGDevice);
    
    return tilesToScroll;
}

short GameContext::ScrollUp(GameLevelPtr theLevel, short tilesToScroll)
{
    // The scrolling functions return the number 
    // of tiles scrolled

	// Bounds checking
	if (vOffset <= kTopEdge)
		return 0;

	// If a full scroll puts us past the top edge
	// then only scroll to the edge.
	if ((vOffset - tilesToScroll) < kTopEdge) {
		short amountToScroll = vOffset;
        return (ScrollUp(theLevel, amountToScroll));
		//return amountToScroll;
	}
		
	CGrafPtr oldPort;
	GDHandle oldGDevice;


	// save previous drawing area
	GetGWorld (&oldPort, &oldGDevice);

	// Set drawing area to the background
	SetGWorld (background->GetBufferStorage(), nil);
	
	// Draw the portion of the background that remains after the scroll
	// For a scroll up, we move the highest rows that remain to
	// the bottom edge of the screen.
	Rect sourceRect;
	Rect destRect;
	
	sourceRect.top = 0;
	sourceRect.left = 0;
	sourceRect.bottom = screenHeight - (tilesToScroll * kTileHeight);
	sourceRect.right = screenWidth;
	
	destRect.top = tilesToScroll * kTileHeight;
	destRect.left = 0;
	destRect.bottom = screenHeight;
	destRect.right = screenWidth;

	Blitter theBlitter;
	theBlitter.SetSourceBuffer(background->GetBufferStorage());
	theBlitter.SetDestinationBuffer(background->GetBufferStorage());
	theBlitter.SetSourceRect(sourceRect);
	theBlitter.SetDestinationRect(destRect);
	theBlitter.SetDrawingMode(srcCopy);
	theBlitter.DrawImageToOffscreenBuffer();
	
	// Update the origin to reflect the scroll
	vOffset = vOffset - tilesToScroll;
			
	// Fill in the new tiles.  For a scroll up, it will be the top rows.
	short columnsInContext = screenWidth / kTileWidth;
	for (short row = 0; row < tilesToScroll; row++) {
		for (short column = 0; column < columnsInContext; column++) {
			DrawTileFromLevelMap(theLevel, row, column);
		}
	}
			
	// restore graphics port and GDevice
	SetGWorld(oldPort, oldGDevice);

    return tilesToScroll;
}

short GameContext::ScrollDown(GameLevelPtr theLevel, short tilesToScroll)
{
    // The scrolling functions return the number 
    // of tiles scrolled

	// Bounds checking.  If we've hit the bottom of the level,
	// don't bother scrolling
	short bottomEdge = theLevel->GetLevelHeight() - (screenHeight / kTileHeight);
	if (vOffset >= bottomEdge)
		return 0;

	// If a full scroll puts us past the bottom edge
	// then only scroll to the edge.
	if ((vOffset + tilesToScroll) > bottomEdge) {
		short amountToScroll = bottomEdge - vOffset;
        return (ScrollDown(theLevel, amountToScroll));    
        //return amountToScroll;	
	}
		
	CGrafPtr oldPort;
	GDHandle oldGDevice;

	//OSStatus error;

	// save previous drawing area
	GetGWorld (&oldPort, &oldGDevice);

	// Set drawing area to the background
	SetGWorld (background->GetBufferStorage(), nil);
	
	// Draw the portion of the background that remains after the scroll
	// For a scroll down, we move the lowest rows that remain to
	// the top edge of the screen.
	Rect sourceRect;
	Rect destRect;
	
	sourceRect.top = tilesToScroll * kTileHeight;
	sourceRect.left = 0;
	sourceRect.bottom = screenHeight;
	sourceRect.right = screenWidth;
	
	destRect.top = 0;
	destRect.left = 0;
	destRect.bottom = screenHeight - (tilesToScroll * kTileHeight);
	destRect.right = screenWidth;
	
	Blitter theBlitter;
	theBlitter.SetSourceBuffer(background->GetBufferStorage());
	theBlitter.SetDestinationBuffer(background->GetBufferStorage());
	theBlitter.SetSourceRect(sourceRect);
	theBlitter.SetDestinationRect(destRect);
	theBlitter.SetDrawingMode(srcCopy);
	theBlitter.DrawImageToOffscreenBuffer();

	// Update the origin to reflect the scroll
	vOffset = vOffset + tilesToScroll;
			
	// Fill in the new tiles.  For a scroll down, it will be
	// the bottom rows
	short rowsInContext = screenHeight / kTileHeight;
	short columnsInContext = screenWidth / kTileWidth;
	
	for (short row = rowsInContext - tilesToScroll; row < rowsInContext; row++) {
		for (short column = 0; column < columnsInContext; column++) {
			DrawTileFromLevelMap(theLevel, row, column);
		}
	}
	
			
	// restore graphics port and GDevice
	SetGWorld(oldPort, oldGDevice);

    return tilesToScroll;
}

// Dirty rectangle functions
void GameContext::AllocateDirtyRectTable(void)
{
	// Assumes we have already determined the screen width and screen height.
	short rowCount = screenHeight / kTileHeight;
	short columnCount = screenWidth / kTileWidth;

    size_t tableSize = rowCount * columnCount * sizeof(char);
	dirtyRectTable = (char*)NewPtr(tableSize);
}

void GameContext::CleanDirtyRectTable(void)
{
	short rowCount = screenHeight / kTileHeight;
	short columnCount = screenWidth / kTileWidth;
	long tableIndex;

	for (short row = 0; row < rowCount; row++) {
		for (short column = 0; column < columnCount; column++) {
			tableIndex = (row * columnCount) + column;
            dirtyRectTable[tableIndex] = kRectangleIsClean;
		}
	}
}

void GameContext::MarkRectangleDirty(short row, short column)
{
	short columnCount = screenWidth / kTileWidth;
	long tableIndex;

    tableIndex = (row * columnCount) + column;
    dirtyRectTable[tableIndex] = kRectangleIsDirty;
}

Boolean GameContext::IsRectangleDirty(short row, short column)
{
	short columnCount = screenWidth / kTileWidth;
	long tableIndex;

    tableIndex = (row * columnCount) + column;

	if (dirtyRectTable[tableIndex] == kRectangleIsDirty)
		return true;
	else
		return false;
}

void GameContext::DisposeDirtyRectTable(void)
{
    if (dirtyRectTable != nil) {
        DisposePtr(dirtyRectTable);
        dirtyRectTable = nil;
    }
}

Rect GameContext::GetDirtyRect(short row, short column)
{
    // This function assumes a dirty rectangle is the size of 
    // one tile.  If you choose a different size, substitute
    // it for the constants kTileHeight and kTileWidth.
    
	Rect result;
	
	result.top = row * kTileHeight;
	result.left = column * kTileWidth;
	result.bottom = result.top + kTileHeight;
	result.right = result.left + kTileWidth;
	
	return result;
}

// Utility functions
Point GameContext::GetGridLocation(short tileNumber)
{
	// Given a tile number from the level map, return
	// the grid location of the tile on the tile board.
	Point result;
	
	result.h = tileNumber % kTilesInRow;
	result.v = tileNumber / kTilesInRow;
	
	return result;
}

Rect GameContext::GetRectFromGridLocation(Point theGridLocation)
{
	Rect result;
	
	result.top = theGridLocation.v * kTileHeight;
	result.left = theGridLocation.h * kTileWidth;
	result.bottom = result.top + kTileHeight; //- 1;
	result.right = result.left + kTileWidth; //- 1;
	
	return result;
}

