// GameApp.cp		Written by Mark Szymczyk
// Book version
// function definitions

#ifndef GAME_APP
	#include "GameApp.h"
#endif

GameApp::GameApp(void)
{
	done = false;
	gameInProgress = false;
    usingHID = false;
	context = nil;
	gameBlitter = nil;
	screenWindow = nil;
 
    player1 = nil;
    player1InputController = nil;
    player1AnimationController = nil;
    player1PhysicsController = nil;
    
    backgroundMusic = nil;
    soundEffect = nil;
    configureControlsDialog = nil;
    
    // At the start we want the whole screen redrawn
    screenScrolled = true;
}

GameApp::~GameApp(void)
{
	CleanUpApp();
}

void GameApp::InitApp(void)
{
	InitToolbox();
	InitAppleEventHandlers();

	SetUpMenuBar();

	gameInProgress = false;
	
	LoadLevel(kLevelResourceID);
	InitializeDrawContext();
	InitializeScreenWindow();
	InitializeGameBlitter();
	InitializePlayer();
	InitializeInputController();
	InitializeSound();
	InitializeWeapons();
	InitializeEnemies(kLevelResourceID);
	
    // If we're running OS X, use the window's
    // back buffer instead of the offscreen buffer
    // as the background.
    Blitter theTileBlitter;
    CGrafPtr screenWindowPort;
    screenWindowPort = GetWindowPort(screenWindow);
    
    if (QDIsPortBuffered(screenWindowPort)) {
        theTileBlitter = context->GetTileBlitter();
        theTileBlitter.SetDestinationBuffer(screenWindowPort);
        gameBlitter->SetDestinationBuffer(screenWindowPort);
    }
    
}

void GameApp::LoadLevel(short levelID)
{
	currentLevel = new GameLevel;
	if (currentLevel == nil)
		ExitToShell();
		
	Boolean success;
	success = currentLevel->ReadLevelData(levelID);
	if (success == false)
		ExitToShell();
        
    currentTileList = new GameTileList;
	if (currentLevel == nil)
		ExitToShell();

    Boolean tilesLoaded;
    tilesLoaded = currentTileList->ReadTilesFromDisk(currentLevel->GetTileID());
    if (!tilesLoaded)
        ExitToShell();

    // Read level population
    /*
    Boolean levelPopulationRead;
    
    levelPopulationRead = ReadLevelPopulation(levelID); 
    if (!levelPopulationRead)
    	ExitToShell();
	*/
}

void GameApp::InitializeScreenWindow(void)
{
	// Create the screen window.  
	screenWindow = GetNewCWindow(kWindowResourceID, nil, WindowPtr(-1));
	if (screenWindow == nil)
		ExitToShell();
	
	// Hide the window initially
	HideWindow(screenWindow);

	// Make the window equal the screen size
	short screenWidth;
	short screenHeight;
	
	// Get the current screen resolution
	GDHandle	mainDevice = NULL;
	mainDevice = GetMainDevice();
	Rect screenRect = (**mainDevice).gdRect;
	screenWidth = screenRect.right - screenRect.left;
	screenHeight = screenRect.bottom - screenRect.top;
	
	
	// Resize the window to match the screen
	Boolean kAddToUpdateRegion = true;
	SizeWindow(screenWindow, screenWidth, screenHeight, kAddToUpdateRegion);

}

void GameApp::InitializeDrawContext(void)
{
	// Create the draw context
	context = new GameContext;
	if (context == nil)
		ExitToShell();

	// Get the current screen resolution
	short screenWidth;
	short screenHeight;
	
	GDHandle	mainDevice = NULL;
	mainDevice = GetMainDevice();
	Rect screenRect = (**mainDevice).gdRect;
	screenWidth = screenRect.right - screenRect.left;
	screenHeight = screenRect.bottom - screenRect.top;
		
	context->SetScreenWidth(screenWidth);
	context->SetScreenHeight(screenHeight);
    
    short currentColorDepth = GetCurrentColorDepth();

	Boolean bufferCreated = context->CreateBackground(screenWidth, screenHeight, 
                currentColorDepth, nil, kNoFlags);
	if (!bufferCreated)	
		ExitToShell();

	// Create tile storage (800 by 640 pixels)
	bufferCreated = context->CreateTileStorage(kStandardWidth, kStandardHeight,
            currentColorDepth, nil, kNoFlags);
	if (!bufferCreated)	
		ExitToShell();
		
	// Draw the tile's pictures into the tile storage
	context->DrawTileStorage(kTilePictureID);	

	// Fill the offscreen buffer with tiles.
	context->DrawBackground(currentLevel);
        
    // Create sprite buffers
    bufferCreated = context->CreateSpriteBuffers(currentColorDepth, nil, kNoFlags);
	if (!bufferCreated)	
		ExitToShell();

    // Draw the sprites into storage
    context->DrawSpriteBuffers();
    
    // Create the dirty rectangle table
    context->AllocateDirtyRectTable();
    context->CleanDirtyRectTable();

}

void GameApp::InitializeGameBlitter(void)
{
	// Create and initialize the blitter
	gameBlitter = new Blitter;
	if (gameBlitter == nil)
		ExitToShell();

	GameOffscreenBufferPtr theBackground = context->GetBackground();
	gameBlitter->Initialize(theBackground->GetBufferStorage(), srcCopy);

}

void GameApp::InitializePlayer(void)
{
    // Create the player, animation, and physics controllers
    player1 = new GamePlayer;
    if (player1 == nil)
        ExitToShell();
	
    player1->SetDestination(context);
    player1->SetSpriteStorage(context->GetPlayerSpriteStorage());
    player1->SetSpriteHeight(kPlayerSpriteHeight);
	player1->SetSpriteWidth(kPlayerSpriteWidth);
	player1->InitializeSpriteBlitter();
	player1->SetMass(kPlayerMass);
  
    Boolean successfulRead = ReadPlayerStartingPosition();
    if (!successfulRead)
        ExitToShell();

    player1AnimationController = new AnimationController;
    if (player1AnimationController == nil)
        ExitToShell();
    player1AnimationController->SetModelToControl(player1);
    
    player1PhysicsController = new PhysicsController;
    if (player1PhysicsController == nil)
        ExitToShell();
    player1PhysicsController->SetModelToControl(player1);

}

void GameApp::InitializeInputController(void)
{
    player1InputController = new InputController;
    if (player1InputController == nil)
    	ExitToShell();
    
    // Check to see if the player has any HID devices connected to his machine
    usingHID = false; //player1InputController->AnyConnectedHIDDevices();
    
    if (usingHID) {
        player1InputController->Initialize();
        player1InputController->Setup();
    }
    
    configureControlsDialog = new ConfigurationDialog;
    if (configureControlsDialog == nil)
        ExitToShell();

}

void GameApp::InitializeSound(void)
{
	// Read the sounds
	GameSong::StartQuickTime();
	
	backgroundMusic = new GameSong;
	if (backgroundMusic == nil)
		return;
	
	Str255 kBackgroundMusicFilename = "\pBackground Music.mov";	
	
    // Reading a movie resource on Mac OS X requires the resource
    // to reside in the data fork.  The only way to do this is to
    // use the Mac OS X version of Resorcerer.  I didn't want to force
    // you to buy Resorcerer so I placed the entire movie in the data
    // fork using QuickTime Pro.  The value moveInDataForkResID tells
    // QuickTime to look in the data fork for the sound.
    backgroundMusic->ReadSoundToPlay(kBackgroundMusicFilename, movieInDataForkResID);

	soundEffect = new GameSound;
	if (soundEffect == nil)
		return;
		
	soundEffect->Create();	
	soundEffect->ReadSoundData(kScreamSoundID);

}

void GameApp::InitializeWeapons(void)
{
    FillWeaponList();
    player1->SetCurrentWeapon(weaponList[kCrossbowIndex]);
    
}

void GameApp::InitializeEnemies(short enemyID)
{
    // Read level population
    Boolean levelPopulationRead;
    
    levelPopulationRead = ReadLevelPopulation(enemyID); 
    if (!levelPopulationRead)
    	ExitToShell();

    enemyAIController = new AIController;
    if (enemyAIController == nil)
    	ExitToShell();	
    	
    enemyAIController->BuildDecisionMaker();
}

Boolean GameApp::ReadPlayerStartingPosition(void)
{
	Vector2D startingPosition;
	startingPosition.SetXComponent(64.0);
	startingPosition.SetYComponent(64.0);
	player1->SetPosition(startingPosition);

    player1->SetWorldX(64);
    player1->SetWorldY(64);
    return true;
    
}

Boolean GameApp::ReadLevelPopulation(short populationID) //GameWeaponPtr weaponList,
        		//GameContextPtr theDrawContext);)
{
	short error;
	//short refNum;
	Handle savedPopulation;
	
	// Clear out anything that's remaining in the monster and item lists
	DeleteEnemyList();
    DeleteItemList();
	
	savedPopulation = GetResource(kLevelPopulationResourceType, populationID);
	error = ResError();
	if ((error != noErr) || (savedPopulation == nil))
		return false;
		
	// Now let's set the level data
	short version;
	short startX;
	short startY;
	short populationCount;
	Size offset = 0;
	
	BlockMoveData(*savedPopulation + offset, &version, sizeof(short));
	offset = offset + sizeof(short);
	
	// If resource is obsolete, don't load it.
	if (version != kCurrentLevelPopulationVersion) {
		ReleaseResource(Handle(savedPopulation));
		//CloseResFile(refNum);
		return false;
	}
	
	// At the start of the level population resource is the
    // player's starting position in the level.  We read this
    // data, but don't do anything with it.
    BlockMoveData(*savedPopulation + offset, &startX, sizeof(short));
	offset = offset + sizeof(short);
	BlockMoveData(*savedPopulation + offset, &startY, sizeof(short));
	offset = offset + sizeof(short);
    
	// Read the number of enemies and items in the level
    BlockMoveData(*savedPopulation + offset, &populationCount, sizeof(short));
	offset = offset + sizeof(short);
	
    LevelResident currentResident;
    PopulationItem whatItIs;
    short x;
    short y;
	
	// Read the enemies and items from disk
    for (short i = 0; i < populationCount; i++) {
		//currentResident = new LevelResident;

		// The amount of data in the LevelResident class is 8 bytes.
		// Unfortunately, sizeof(LevelResident) returns 12 bytes.
		// I had to read the elements of the LevelResident class
        // separately to read the enemies from disk properly.
		BlockMoveData(*savedPopulation + offset, &whatItIs, sizeof(PopulationItem));
        currentResident.SetResident(whatItIs);
        offset = offset + sizeof(PopulationItem);
        
        BlockMoveData(*savedPopulation + offset, &x, sizeof(short));
        currentResident.SetStartX(x);
        offset = offset + sizeof(short);
        
        BlockMoveData(*savedPopulation + offset, &y, sizeof(short));
        currentResident.SetStartY(y);
        offset = offset + sizeof(short);
        
        if (IsResidentAnEnemy(&currentResident)) {
			AddEnemyToList(&currentResident, weaponList, context);
		}
		else {
			AddItemToList(&currentResident, context);
		}
		
	}
	
	// Clean up
	ReleaseResource(Handle(savedPopulation));
	error = ResError();
	if (SeriousError(error))
		return false;

	return true;
	
}

void GameApp::InitToolbox (void)
{
	// Initialize all the needed managers. 
	// In Carbon, you need to initialize only the cursor.
	
	InitCursor();
	FlushEvents(everyEvent,0);
		
}

void GameApp::InitAppleEventHandlers (void)
{
	OSErr error;
	
	// Install the handler to handle the Quit Application Apple event.
	// We need to handle this event to let the user quit in OS X.
	error = AEInstallEventHandler(kCoreEventClass, kAEQuitApplication,  
								NewAEEventHandlerUPP(HandleQuitApplication), (long) this, false);

}

void GameApp::CleanUpApp(void)
{
    soundEffect->Dispose();
    DisposeWindow(screenWindow);
	
	// Release the memory of the variables we allocated
	if (context != nil) {
        delete context;
        context = nil;
    }
    
	if (currentLevel != nil) {
        delete currentLevel;
        currentLevel =  nil;
    }
    
	if (currentTileList != nil) {
        delete currentTileList;
        currentTileList = nil;
    }
    
	if (gameBlitter != nil) { 
        delete gameBlitter;
        gameBlitter = nil;
    }
    
	if (player1 != nil) {
        delete player1;
        player1 = nil;
    }
	
    if (player1InputController != nil) {
        delete player1InputController;
        player1InputController = nil;
    }
    
    if (configureControlsDialog != nil) {
        delete configureControlsDialog;
        configureControlsDialog = nil;
    }
    
	if (player1AnimationController != nil) {
        delete player1AnimationController;
        player1AnimationController = nil;
    }
    
	if (player1PhysicsController != nil) {
        delete player1PhysicsController;
        player1PhysicsController = nil;
    }

	if (enemyAIController != nil) {
        delete enemyAIController;
        enemyAIController = nil;
    }

	if (backgroundMusic != nil) {
		delete backgroundMusic;
		backgroundMusic = nil;
	}
	
	if (soundEffect != nil) {
		delete soundEffect;
		soundEffect = nil;
	}
	
	GameSong::ExitQuickTime();
    
    DeleteMissileList();
	DeleteEnemyList();
	DeleteItemList();

}

void GameApp::RemoveAppleEventHandlers(void)
{
	OSErr error;
	
	error = AERemoveEventHandler(kCoreEventClass, kAEQuitApplication,  
							NewAEEventHandlerUPP(HandleQuitApplication), false);

}

// event related functions

void GameApp::EventLoop(void)
{
	RgnHandle	cursorRgn = nil;
	Boolean		haveEvent;
	EventRecord	event;
	
	do{
		// If we're in full screen mode, run the game loop.
		if (gameInProgress) {
			GameLoop();
		}
		else {
			// Perform the traditional Mac OS event loop.
			haveEvent = WaitNextEvent(everyEvent, &event, kSleepValue, cursorRgn);
		
			if (haveEvent) {
				HandleEvent(&event);
			}
			else {
				HandleIdleEvent();
			}
		}
	} while (! done);
}

void GameApp::HandleIdleEvent(void)
{
	// do nothing in base class
}

void GameApp::HandleEvent (EventRecord* event)
{

	switch (event->what) {
		
		case mouseDown:
			HandleMouseDown(event);
			break;
			
		case mouseUp:
			HandleMouseUp(event);
			break;
			
		case keyDown:
		case autoKey:
			CheckForCommandKey(event);
			break;
			
		case activateEvt:
			//HandleActivate(event);
			break;
			
		case updateEvt:
			HandleUpdate(event);
			break;
			
		case diskEvt:
			HandleDiskEvent(event);
			break;
			
		case osEvt:
			HandleOSEvent(event);
			break;

		case kHighLevelEvent:
			HandleHighLevelEvent(event);
			break;
			
	}
}

void GameApp::HandleMouseDown(EventRecord* event)
{
	int 		partOfScreen;
	WindowPtr 	thisWindow;

    if (IsDialogEvent(event)) {
        configureControlsDialog->DoContent(event);
    }
		
	// Map cursor location at time of mouse down event
	partOfScreen = FindWindow(event->where, &thisWindow);
	
	switch(partOfScreen){
		case inMenuBar:
			AdjustMenus();
			HandleMenuCommand(MenuSelect(event->where));
			break;
			
		case inContent:
			// If you wanted a mouse click in the content
			// area of the window to do something special,
			// you would put it here.
			
			break;

        // This code lets the player move the 
        // controller configuration dialog box
        case inDrag:
            if(configureControlsDialog != nil)
                configureControlsDialog->Drag(event);
					
		// The game has just one plain full screen window 
		// so there's no need to check for the
		// drag, grow, close and zoom areas of windows.	
	}
}

void GameApp::HandleMouseUp(EventRecord* event)
{
	// Do nothing in a Game class
}

void GameApp::CheckForCommandKey (EventRecord* event)
{
    // Checks if Command Key was pressed.  If so, handle menu selection.

    //WindowPtr activeWindow = nil;

    if (IsDialogEvent(event)) {
        configureControlsDialog->HandleKeyDown(event);
    }
	
    char key;
	
    key = (char)(event->message & charCodeMask); // get the key pressed 
    
    // We need this additional code on Mac OS X
    // to quit the game that explicitly checks for 
    // the Command Q combination.
    Boolean cmdKeyHit;
    if (event->modifiers & cmdKey)
        cmdKeyHit = true;
    else
        cmdKeyHit = false;
    
    // Quit on Command Q
    if ((key == kLetterQ) && (cmdKeyHit))
        Quit();
        
    // Handle other menu selection
    if (cmdKeyHit) {		
        AdjustMenus();
        HandleMenuCommand(MenuKey(key));
    }
}  		

	
void GameApp::HandleUpdate (EventRecord* event)
{
    if (IsDialogEvent(event)) {
        configureControlsDialog->Update();
    }
	
	WindowPtr theWindow;
		
	theWindow = (WindowPtr) event->message;

	// This program has only one window so we know that
	// theWindow is screenWindow.  If the screen window
	// is visible, we redraw the window's contents.
	// When the user pauses the program, I hide the screen 
	// window, which generates an update event.  If I update
	// the window in this case, Spotlight gives me a
	// "Current GrafPort isn't window for EndUpdate" error.
	
	if (IsWindowVisible(theWindow)) {
		BeginUpdate(theWindow);
		RenderFrame();
		EndUpdate(theWindow);
	}
	
	
}	


void GameApp::HandleDiskEvent (EventRecord* event)
{
	// Carbon doesn't accept disk inserted
	// events.  If you were writing a version of
	// your game for Mac OS 8/9 only, you would
	// remove the comments below to handle the
	// disk inserted event.
	
	/*
	Point	dummyPoint;
	OSErr	err;
	
	dummyPoint.h = 0;
	dummyPoint.v = 0;
	
	if (HiWord(event->message) != noErr) {
		DILoad;
		err = DIBadMount(dummyPoint, event->message);
		DIUnload;
	}
	// Do nothing if disk was inserted correctly
	*/
}


void GameApp::HandleOSEvent(EventRecord* event)
{
	char	eventType;

	eventType = event->message >> 24; /* high byte of message field */

	if (eventType & mouseMovedMessage)	 /* mouse moved event */
	{
		HandleMouseMoved(event);
	}
	
	else if (eventType & suspendResumeMessage)	/* suspend/resume event */
	{
		HandleSuspendResume(event);
	}

}

void GameApp::HandleHighLevelEvent(EventRecord* event)
{
	OSErr err;
	
	// base class just processes the required Apple Events
	err = AEProcessAppleEvent(event);
	
}

OSErr GameApp::GotRequiredAEParameters(const AppleEvent* theAppleEvent)
{
	DescType returnedType;
	Size actualSize;
	OSErr err;
	
	err = AEGetAttributePtr(theAppleEvent, keyMissedKeywordAttr, typeWildCard, &returnedType, nil, 0,
							&actualSize);
	
	if (err == errAEDescNotFound)
		return noErr;
	else
		return errAEParamMissed;
		
}

pascal OSErr GameApp::HandleQuitApplication(const AppleEvent* theAppleEvent, AppleEvent* theReply, long userData)
{	
	// just quit as if the user selected Quit from the File Menu
	// in Mac OS 8/9.  We need this function so the user can
	// quit from the Application menu in OS X.
	
	GameAppPtr currentApp = (GameAppPtr)userData;
	currentApp->Quit();
	
	return GotRequiredAEParameters(theAppleEvent);
}

void GameApp::HandleMouseMoved(EventRecord* event)
{
	// treat mouse moved like an idle event
	HandleIdleEvent();
}

void GameApp::HandleSuspendResume(EventRecord* event)
{
	
}

// menu related functions

void GameApp::SetUpMenuBar()
{
	Handle menuBar;
	
	menuBar = GetNewMBar(kMenuBarID);
	if (menuBar == nil)
		return;
		
	SetMenuBar(menuBar);
	DisposeHandle(menuBar);
	
	// Add items to Apple menu.
	// Carbon does it automatically.
	//AppendResMenu(GetMenuHandle(kAppleMenu), 'DRVR');
	
	DrawMenuBar();
}

void GameApp::AdjustMenus(void)
{
	AdjustApplicationMenu();
    AdjustFileMenu();
    AdjustOptionsMenu();
}

void GameApp::AdjustApplicationMenu(void)
{
    // Mac OS X has an application menu where the user
    // quits the application.  We need to append the
    // Cmd-Q keyboard equivalent to the Quit menu item.
    
    OSStatus error;
    MenuRef appMenu;
    MenuItemIndex quitMenuItem;
    
    // Find the Quit menu item.  OS X supplies it for us so we
    // don't know the resource ID or the menu item of the
    // Quit menu item.
    error = GetIndMenuItemWithCommandID(nil, kHICommandQuit, 1, &appMenu, &quitMenuItem);
    if (error != noErr)
        return;
    
    // Add the Cmd-Q text to the Quit menu item.  By default
    // OS X does not assign a command key equivalent.
    // Using the constant kMenuNoModifiers means to use
    // the command key plus the key you want (Q in this case)
    error = SetMenuItemCommandKey(appMenu, quitMenuItem, kUseVirtualKeyCode, 
                            kMenuNoModifiers + kQuitKey);
    
}

void GameApp::AdjustFileMenu(void)
{
	// If the player is running Mac OS X, we should disable the Quit
	// menu item in the File Menu.
	Boolean runningOSX;
	SInt32 OSVersion;
	OSErr error;
	
	error = Gestalt(gestaltSystemVersion, &OSVersion);
	if (error != noErr)
		return;
		
	if (OSVersion >= 0x01000)
		runningOSX = true;
	else
		runningOSX = false;
		
	MenuRef menu;
	
	if (runningOSX) {
		menu = GetMenuHandle(kFileMenu);
		DisableMenuItem(menu, kQuit);
	}
	
}
void GameApp::AdjustOptionsMenu(void)
{
    MenuRef menu;
    menu = GetMenuHandle(kOptionsMenu);
    
    if(usingHID)
        EnableMenuItem(menu, kConfigureMenuItem);
    else
        DisableMenuItem(menu, kConfigureMenuItem);
}

void GameApp::HandleMenuCommand(long menuResult)
{
	int menuID;
	int menuItem;
	
	menuID = HiWord(menuResult);
	menuItem = LoWord(menuResult);
	
	switch (menuID) {
		case kAppleMenu:
			HandleAppleCommand(menuItem);
			break;
		
		case kFileMenu:
			HandleFileCommand(menuItem);
			break;

		case kOptionsMenu:
			HandleOptionsCommand(menuItem);
			break;
		
	}
	
}

void GameApp::HandleAppleCommand (int menuItem)
{
	//Str255 itemName;
	//int refNum;
	
	if (menuItem == kAbout)
		DisplayAboutBox(kAboutBoxID);
	/*
	else {
		// Carbon opens the Apple menu items automatically
		GetMenuItemText(GetMenuHandle(kAppleMenu), menuItem, itemName);
		refNum = OpenDeskAcc(itemName);
	}
	*/
}

void GameApp::DisplayAboutBox(short resourceID)
{
	//short result = Alert(resourceID, nil);
}

void GameApp::HandleFileCommand (int menuItem)
{
	switch (menuItem) {
		case kResume:
			Resume();
			break;
			
		case kQuit:
			Quit();
			break;
			
		default:
			break;
		
	}
	
}

void GameApp::HandleOptionsCommand (int menuItem)
{
	switch (menuItem) {
		case kConfigureMenuItem:
			ConfigureControls();
			break;
						
		default:
			break;
		
	}
	
}

void GameApp::ConfigureControls(void)
{
    if (usingHID) {
        configureControlsDialog->SetCurrentController(player1InputController);
        configureControlsDialog->Create(kConfigurationDialogID);
        configureControlsDialog->Show();
    }
}

void GameApp::Quit(void)
{
	// Set done to true and quit program
	done = true;	
}

void GameApp::ChangeResumeMenuItemText(void)
{
	// In this program, I want to change the text of
	// the first menu item from Start to Resume after the
	// player chooses Start from the menu.
	
	MenuHandle theMenu;
	StringHandle theNewText;
	
	theMenu = GetMenuHandle(kFileMenu);
	
	// Read the new text from the resource on disk.
	theNewText = GetString(kResumeTextID);
	
	// SetMenuItemText takes in a value of Str255 which
	// is a string pointer.  theNewText is a handle so we
	// have to pass in (*theNewText), which is a pointer.
	SetMenuItemText(theMenu, kResume, (*theNewText));
	
	// Dispose of the string handle to avoid a memory leak.
	ReleaseResource(Handle(theNewText));
}

// Game related functions
void GameApp::GameLoop(void)
{
    // Calculate the elapsed time since the last frame.
    // timeStep is a data member of the GameApp class.
    timeStep = CalculateTimeStep();
    
	InputControllerAction playerAction;
	InputControllerAction playerMovement;
    //InputControllerAction keyboardAction;
    
    Boolean escapeKeyPressed;
    
    if (usingHID) {
        playerAction = player1InputController->DetermineAction();
        playerMovement = player1InputController->DetermineAnalogMovement();
        
        escapeKeyPressed = CheckForEscapeKey();
        if (escapeKeyPressed) {
            playerAction = kPauseGame;
        }
    }
    else {
        playerAction = ReadKeyboard();
    }
    
    /*
    if ((!usingHID) || (keyboardAction == kPauseGame)) {
        playerAction = keyboardAction;
    }
    */
    
	AnimatePlayer(playerAction);
    if (usingHID)
        AnimatePlayer(playerMovement);
 
    AnimateEnemies();
    AnimateMissiles();
    CollisionDetection();
    
    PlayMusic();
    ScrollBackground();
	RenderFrame();
    
    // Slow down game to 60 frames per second
    // If the program is running slow on your computer,
    // comment these lines out.
    //UInt32 delayAmount = 1;
    //UInt32 actualDelay;
    //Delay(delayAmount, &actualDelay);
}

Boolean GameApp::CheckForEscapeKey(void)
{
    KeyMap currentKeyboardState;
    GetKeys(currentKeyboardState);
    
    Boolean result;
    result = WasKeyPressed(kEscapeKey, currentKeyboardState);
    
    return result;
}

InputControllerAction GameApp::ReadKeyboard(void)
{
    KeyMap currentKeyboardState;
    GetKeys(currentKeyboardState);
    
    Boolean movedUp = false;
    Boolean movedDown = false;
    Boolean movedLeft = false;
    Boolean movedRight = false;
 
    if (WasKeyPressed(kEscapeKey, currentKeyboardState)) {
        player1InputController->SetXAxisValue(kAxisCenterValue);
        player1InputController->SetYAxisValue(kAxisCenterValue);
        return kPauseGame;
    }
    else if (WasKeyPressed (kSpaceBar, currentKeyboardState)) {
        player1InputController->SetXAxisValue(kAxisCenterValue);
        player1InputController->SetYAxisValue(kAxisCenterValue);
        return kAttack;
    }
        
    if (WasKeyPressed(kUpArrow, currentKeyboardState)){
        movedUp = true;
        player1InputController->SetYAxisValue(kAxisMinimumValue);
    }
    
    if (WasKeyPressed(kDownArrow, currentKeyboardState)){
        movedDown = true;
        player1InputController->SetYAxisValue(kAxisMaximumValue);
    }
    
    if (WasKeyPressed(kLeftArrow, currentKeyboardState)){
        movedLeft = true;
        player1InputController->SetXAxisValue(kAxisMinimumValue);
    }
    
    if (WasKeyPressed(kRightArrow, currentKeyboardState)) {
        movedRight = true;
        player1InputController->SetXAxisValue(kAxisMaximumValue);
    }

    if ((!movedUp) && (!movedDown)) {
        player1InputController->SetYAxisValue(kAxisCenterValue);
    }

    if ((!movedLeft) && (!movedRight)) {
        player1InputController->SetXAxisValue(kAxisCenterValue);
    }
    
    // Determine direction of movement
	if ((movedUp) && (movedLeft))
		return kMoveUpAndLeft;
	else if ((movedUp) && (movedRight))
		return kMoveUpAndRight;
	else if ((movedDown) && (movedLeft))
		return kMoveDownAndLeft;
	else if ((movedDown) && (movedRight))
		return kMoveDownAndRight;

	// At this point, we know the movement is not diagonal
	else if (movedUp)
		return kMoveUp;
	else if (movedDown)
		return kMoveDown;
	else if (movedLeft)
		return kMoveLeft;
	else if (movedRight) 
		return kMoveRight;
	else	
		return kNoMovement;

}

void GameApp::AnimatePlayer(InputControllerAction playerAction)
{
    /*
    Vector2D forceVector = CalculatePlayerForce();
    
    //double timeStep = CalculateTimeStamp();
    
    player1PhysicsController->UpdatePhysics(playerAction, &forceVector, timeStep, 
    		currentLevel, currentTileList);
    
    //player1PhysicsController->UpdatePhysics(playerAction, currentLevel, currentTileList);
	player1AnimationController->UpdateAnimation(playerAction);
    */
    
    GameWeapon playerWeapon = player1->GetCurrentWeapon();
    
    switch (playerAction) {
        case kAttack:
            if (player1->CanAttack()) {
                // Update the ammo count to reflect the attack
                player1->FireMissile();
                // Now, actually launch the missile
                LaunchMissile(player1, &playerWeapon);
            }
            break;
            
        case kPauseGame:
            Pause();
            break;
            
        default:
            break;
    }

    Vector2D forceVector = CalculatePlayerForce();
    
    player1PhysicsController->UpdatePhysics(playerAction, &forceVector, timeStep, 
    		currentLevel, currentTileList);
    
	player1AnimationController->UpdateAnimation(playerAction);
    NotifyEnemiesOfIntruders();
}

void GameApp::AnimateEnemies(void)
{
	Iterator enemyListIterator(&enemyList);
	GameEnemyPtr currentEnemy = nil;
	InputControllerAction enemyAction;

	short playerCenterX = player1->GetWorldX() + (kPlayerSpriteWidth / 2);
	short playerCenterY = player1->GetWorldY() + (kPlayerSpriteHeight / 2);

	do {
		// listIterator.Next provides us the link to the current missile
		currentEnemy = (GameEnemyPtr)enemyListIterator.Next();
		if (currentEnemy != nil) {
			enemyAIController->SetModelToControl(currentEnemy);
			enemyAction = enemyAIController->PerformManeuver(playerCenterX, playerCenterY,
						currentLevel, currentTileList);
			MoveEnemy(currentEnemy, enemyAction);
		}
	} while (currentEnemy != nil);

}

void GameApp::MoveEnemy(GameEnemyPtr enemy, InputControllerAction theAction)
{
	PhysicsController enemyPhysicsController;
	enemyPhysicsController.SetModelToControl(enemy);

	AnimationController enemyAnimationController;
	enemyAnimationController.SetModelToControl(enemy);
    
    Vector2D forceVector = enemy->CalculateForce(theAction);

	// Update the dirty rectangle table before movement to
	// prevent a problem where the enemy sprite remains on the
	// screen when the enemy moves off the screen.
	if (enemyAnimationController.IsInScreenArea()) {
		enemy->UpdateDirtyRectTable();
	}

	enemyPhysicsController.UpdatePhysics(theAction, &forceVector, timeStep, 
				currentLevel, currentTileList);

	enemyAnimationController.UpdateAnimation(theAction);
}

void GameApp::AnimateMissiles(void)
{
	Iterator listIterator(&missileList);
	GameWeaponPtr currentMissile = nil;

	do {
		// listIterator.Next provides us the link to the current missile
		currentMissile = (GameWeaponPtr)listIterator.Next();
		if (currentMissile != nil) {
			MoveMissile(currentMissile);
		}
	} while (currentMissile != nil);

}

void GameApp::MoveMissile(GameWeaponPtr theMissile)
{
	//short xVelocity;
	//short yVelocity;

    AnimationController missileAnimationController;
    missileAnimationController.SetModelToControl(theMissile);
    
	PhysicsController missilePhysicsController;
	missilePhysicsController.SetModelToControl(theMissile);
    
    theMissile->CalculateCurrentForce(timeStep);
    Vector2D forceVector = theMissile->GetCurrentForce();
    
	switch(theMissile->GetSpriteDirection()) {
		case kSpriteFacingUp:
			missilePhysicsController.UpdatePhysics(kMoveUp, &forceVector,
                    timeStep, currentLevel, currentTileList);
            missileAnimationController.UpdateAnimation(kMoveUp);
			break;
				
		case kSpriteFacingDown:
			missilePhysicsController.UpdatePhysics(kMoveDown, &forceVector, 
                    timeStep, currentLevel, currentTileList);
            missileAnimationController.UpdateAnimation(kMoveDown);
			break;

		case kSpriteFacingLeft:
			missilePhysicsController.UpdatePhysics(kMoveLeft, &forceVector,
                    timeStep, currentLevel, currentTileList);
            missileAnimationController.UpdateAnimation(kMoveLeft);
			break;
			
		case kSpriteFacingRight:
			missilePhysicsController.UpdatePhysics(kMoveRight, &forceVector,
                    timeStep, currentLevel, currentTileList);
            missileAnimationController.UpdateAnimation(kMoveRight);
			break;
			
		default:
			break;
	}

	// If missile stopped moving, kill it
    if (theMissile->StoppedMoving())
        KillMissile(theMissile);
}	

void GameApp::CollisionDetection(void)
{
	Iterator enemyListIterator(&enemyList);
	GameEnemyPtr currentEnemy = nil;
	PhysicsController enemyPhysicsController;

	// Check for collision between missiles and monsters and missiles and the player.
	Iterator listIterator(&missileList);
	GameWeaponPtr currentMissile = nil;
	PhysicsController missilePhysicsController;

	do {
		// listIterator.Next provides us the link to the current missile
		currentMissile = (GameWeaponPtr)listIterator.Next();
		if (currentMissile != nil) {
			missilePhysicsController.SetModelToControl(currentMissile);
			
			do {
				currentEnemy = (GameEnemyPtr)enemyListIterator.Next();
				if (currentEnemy != nil) {
					// enemyPhysicsController is declared earlier in the function
					enemyPhysicsController.SetModelToControl(currentEnemy);
					if (missilePhysicsController.DetectCollision(currentEnemy)) {
						//soundEffect->Play();
						//KillMonster(currentEnemy);
						//currentEnemy->TakeHit(currentMissile);
						enemyList.RemoveItem(currentEnemy);
                        delete currentEnemy;
                        currentEnemy = nil;
						KillMissile(currentMissile);
					}
				}
			} while (currentEnemy != nil);					
			
			// Check for collision with player
			/*
            if (missilePhysicsController.DetectCollision(player1)) {
				//if (soundEffect->IsSoundPlaying() == false)
				//	soundEffect->Play();
				player1->TakeHit(currentMissile);
				KillMissile(currentMissile);
			}
			*/
            
			/*
			for (short i = 0; i < numberOfMonsters; i++) {
				if (monsterList[i].WasKilled() == false) {
					if (currentMissile->DetectCollision(monsterList[i])) {
						soundEffect->Play();
						monsterList[i].Kill();
					}
				}
			}
			*/
		}
		
	} while (currentMissile != nil);

}

void GameApp::PlayMusic(void)
{
	backgroundMusic->PlayRepeatedly();

    // You would use this code if you used
    // the Sound Manager to play the music
    // instead of QuickTime.
    
	//if (backgroundMusic->IsSoundPlaying() == false)
	//	backgroundMusic->Play();
}

void GameApp::ScrollBackground(void)
{
	short playerScreenTop = player1->GetWorldY() - (context->GetVOffset() * kTileHeight);
	short playerScreenBottom = playerScreenTop + player1->GetSpriteHeight();
	short playerScreenLeft = player1->GetWorldX() - (context->GetHOffset() * kTileWidth);
	short playerScreenRight = playerScreenLeft + player1->GetSpriteWidth();
	
	// If wer're too close to the edge of the screen, scroll
	// in the appropriate direction.
	
	short scrollBoundaryTop = kTileHeight * 2;
	short scrollBoundaryLeft = kTileWidth * 2;
	
    short tilesScrolledUp = 0;
	if (playerScreenTop < scrollBoundaryTop) {
		tilesScrolledUp = context->ScrollUp(currentLevel, kEightTiles);
	}
	
    short tilesScrolledLeft = 0;
	if (playerScreenLeft < scrollBoundaryLeft) {
		tilesScrolledLeft = context->ScrollLeft(currentLevel, kEightTiles);
	}

	short scrollBoundaryBottom = context->GetScreenHeight() - (kTileHeight * 2);
	short scrollBoundaryRight = context->GetScreenWidth() - (kTileWidth * 2);
		
    short tilesScrolledDown = 0;
	if (playerScreenBottom > scrollBoundaryBottom) {
		tilesScrolledDown = context->ScrollDown(currentLevel, kEightTiles);
	}
		
    short tilesScrolledRight = 0;
	if (playerScreenRight > scrollBoundaryRight) {
		tilesScrolledRight = context->ScrollRight(currentLevel, kEightTiles);
	}

    // Test if we actually scrolled.  This fixes a problem
    // where fragments of the sprite do not erase when the 
    // sprite approaches the edge of the level.
    if (tilesScrolledUp > 0)
        screenScrolled = true;
    else if (tilesScrolledDown > 0)
        screenScrolled = true;
    else if (tilesScrolledLeft > 0)
        screenScrolled = true;
    else if (tilesScrolledRight > 0)
        screenScrolled = true;
        
}
		
void GameApp::RenderFrame(void) 
{
    if (screenScrolled) {
        // Redraw the whole screen because a scroll will
        // mark all rectangles as dirty.  One big draw is 
        // faster than drawing the whole screen one tile at a time.
        RenderScreen();
    }
    else {
        RenderDirtyRectangles();
    }
    
}

void GameApp::RenderScreen(void) 
{
	// Get the current screen resolution
	GDHandle	mainDevice = nil;
	mainDevice = GetMainDevice();
	Rect screenRect = (**mainDevice).gdRect;

    DrawMissilesOffscreen();
    DrawEnemiesOffscreen();
    DrawPlayerOffscreen();    
    
    // Draw from the offscreen buffer to the screen
    gameBlitter->Setup(GetWindowPort(screenWindow), screenRect, screenRect);
    gameBlitter->DrawImageToScreen();
    gameBlitter->FlushBackBuffer();
    screenScrolled = false;
    context->CleanDirtyRectTable();
}

void GameApp::RenderDirtyRectangles(void) 
{
	short rowCount = context->GetScreenHeight() / kTileHeight;
	short columnCount = context->GetScreenWidth() / kTileWidth;
    Rect dirtyRect;
    
	for (short row = 0; row < rowCount; row++) {
		for (short column = 0; column < columnCount; column++) {
			if (context->IsRectangleDirty(row, column)) {
                // Draw the tile to the offscreen buffer
                context->DrawTileFromLevelMap(currentLevel, row, column);
                
            }
		}
	}
    
    DrawMissilesOffscreen();
    DrawEnemiesOffscreen();
    DrawPlayerOffscreen();

    for (short row = 0; row < rowCount; row++) {
		for (short column = 0; column < columnCount; column++) {
			if (context->IsRectangleDirty(row, column)) {
                // Draw the tile to the screen
                dirtyRect = context->GetDirtyRect(row, column);
                gameBlitter->Setup(GetWindowPort(screenWindow), dirtyRect, dirtyRect);
                gameBlitter->DrawImageToScreen();
            }
		}
	}
    
    DrawMissilesToScreen();
    DrawEnemiesToScreen();
    DrawPlayerToScreen();
    gameBlitter->FlushBackBuffer();
    
    // We did all our drawing so we can clean the table
    context->CleanDirtyRectTable();
    
}

void GameApp::DrawMissilesOffscreen(void)
{
	//GameOffscreenBufferPtr theBackground = context->GetBackground();
    
	// Draw missiles that are on the screen
	Iterator listIterator(&missileList);
	GameWeaponPtr currentMissile = nil;

    AnimationController missileAnimationController;
    Blitter theTileBlitter = context->GetTileBlitter();
    CGrafPtr destination = theTileBlitter.GetDestinationBuffer();
    
	do {
		// listIterator.Next provides us the link to the current missile
		currentMissile = (GameWeaponPtr)listIterator.Next();
		missileAnimationController.SetModelToControl(currentMissile);
        if (currentMissile != nil) {
			
			if (missileAnimationController.IsInScreenArea()) { 
				currentMissile->DetermineSpriteToDraw();
				currentMissile->SetupSpriteBlitter(destination);
                //currentMissile->SetupSpriteBlitter(theBackground->GetBufferStorage());
				currentMissile->Draw();
			}
			else {
				KillMissile(currentMissile);
			}
			
		}
	} while (currentMissile != nil);
}

void GameApp::DrawMissilesToScreen(void)
{    
	// Draw missiles that are on the screen
	Iterator listIterator(&missileList);
	GameWeaponPtr currentMissile = nil;

    AnimationController missileAnimationController;
    
	do {
		// listIterator.Next provides us the link to the current missile
		currentMissile = (GameWeaponPtr)listIterator.Next();
		missileAnimationController.SetModelToControl(currentMissile);
        if (currentMissile != nil) {
			
			if (missileAnimationController.IsInScreenArea()) { 
				DrawMissileOnScreen(currentMissile);
			}
			else {
				KillMissile(currentMissile);
			}
			
		}
	} while (currentMissile != nil);
}

void GameApp::DrawMissileOnScreen(GameWeaponPtr missileToDraw)
{
    // Draw it to the screen
    Rect missileRect;
    
    missileRect.top = missileToDraw->GetWorldY() - (context->GetVOffset() * kTileHeight);
    missileRect.bottom = missileRect.top + missileToDraw->GetSpriteHeight();
    missileRect.left = missileToDraw->GetWorldX() - (context->GetHOffset() * kTileWidth);
    missileRect.right = missileRect.left + missileToDraw->GetSpriteWidth();
    
    // Draw from the offscreen buffer to the screen
    gameBlitter->Setup(GetWindowPort(screenWindow), missileRect, missileRect);
    gameBlitter->DrawImageToScreen();
}

void GameApp::DrawEnemiesOffscreen(void)
{
	//GameOffscreenBufferPtr theBackground = context->GetBackground();
    
	// Draw missiles that are on the screen
	Iterator listIterator(&enemyList);
	GameEnemyPtr currentEnemy = nil;

    AnimationController enemyAnimationController;
    Blitter theTileBlitter = context->GetTileBlitter();
    CGrafPtr destination = theTileBlitter.GetDestinationBuffer();
    
	do {
		// listIterator.Next provides us the link to the current enemy
		currentEnemy = (GameEnemyPtr)listIterator.Next();
		enemyAnimationController.SetModelToControl(currentEnemy);
        if (currentEnemy != nil) {
			
			if (enemyAnimationController.IsInScreenArea()) { 
				currentEnemy->DetermineSpriteToDraw();
				currentEnemy->SetupSpriteBlitter(destination);
                //currentEnemy->SetupSpriteBlitter(theBackground->GetBufferStorage());
				currentEnemy->Draw();
			}
			
		}
	} while (currentEnemy != nil);
}

void GameApp::DrawEnemiesToScreen(void)
{    
	// Draw enemies that are on the screen
	Iterator listIterator(&enemyList);
	GameEnemyPtr currentEnemy = nil;

    AnimationController enemyAnimationController;
    
	do {
		// listIterator.Next provides us the link to the current enemy
		currentEnemy = (GameEnemyPtr)listIterator.Next();
		enemyAnimationController.SetModelToControl(currentEnemy);
        if (currentEnemy != nil) {
			
			if (enemyAnimationController.IsInScreenArea()) { 
				DrawEnemyOnScreen(currentEnemy);
			}
			
		}
	} while (currentEnemy != nil);
}

void GameApp::DrawEnemyOnScreen(GameEnemyPtr enemyToDraw)
{
    // Draw it to the screen
    Rect enemyRect;
    
    enemyRect.top = enemyToDraw->GetWorldY() - (context->GetVOffset() * kTileHeight);
    enemyRect.bottom = enemyRect.top + enemyToDraw->GetSpriteHeight();
    enemyRect.left = enemyToDraw->GetWorldX() - (context->GetHOffset() * kTileWidth);
    enemyRect.right = enemyRect.left + enemyToDraw->GetSpriteWidth();
    
    // Draw from the offscreen buffer to the screen
    gameBlitter->Setup(GetWindowPort(screenWindow), enemyRect, enemyRect);
    gameBlitter->DrawImageToScreen();
}

void GameApp::DrawPlayerOffscreen(void)
{
	//GameOffscreenBufferPtr theBackground = context->GetBackground();
    
    if (player1 == nil)
        return;
        
    // Draw player sprite offscreen
    player1->DetermineSpriteToDraw();    

    Blitter theTileBlitter = context->GetTileBlitter();
    CGrafPtr destination = theTileBlitter.GetDestinationBuffer();
    player1->SetupSpriteBlitter(destination);
    //player1->SetupSpriteBlitter(theBackground->GetBufferStorage());
    player1->Draw();

}

void GameApp::DrawPlayerToScreen(void)
{
    // Draw it to the screen
    Rect playerRect;
    
    playerRect.top = player1->GetWorldY() - (context->GetVOffset() * kTileHeight);
    playerRect.bottom = playerRect.top + player1->GetSpriteHeight();
    playerRect.left = player1->GetWorldX() - (context->GetHOffset() * kTileWidth);
    playerRect.right = playerRect.left + player1->GetSpriteWidth();
    
    // Draw from the offscreen buffer to the screen
    gameBlitter->Setup(GetWindowPort(screenWindow), playerRect, playerRect);
    gameBlitter->DrawImageToScreen();
}

void GameApp::Pause(void)
{
	// When the user pauses the game, we want to return
	// to normal Mac OS mode.  This involves hiding the screen
	// window, redrawing the menu bar and showing the mouse cursor.

    if (usingHID)
    	player1InputController->Pause();
        
	StopFullScreen();
	HideWindow(screenWindow);
	ShowCursor();
	ChangeResumeMenuItemText();
	DrawMenuBar();
	gameInProgress = false;
	backgroundMusic->Pause();
}

void GameApp::StopFullScreen(void)
{
	OSErr error;
	
	error = EndFullScreen(quickTimeStateData, nil);
}

void GameApp::Resume(void)
{
	// We want to go into game mode.  To do this, we
	// go into full screen mode, hide the menu bar and
	// hide the mouse cursor.
    if (usingHID)
    	player1InputController->Resume();
	
	StartFullScreen();
	ShowWindow(screenWindow);
	HideCursor();
	gameInProgress = true;
    screenScrolled = true;
    backgroundMusic->Resume();
    RenderFrame();
    
    // It's important to calculate the time step so
    // the physics engine works properly.  When starting
    // the game or after a long pause, the time step will
    // be large.  If it is too big, some enemy sprites will
    // not appear on the screen.
    timeStep = CalculateTimeStep();
}

void GameApp::StartFullScreen(void)
{
	OSErr error;

	// Set the erase color to black.
	// You could change this to a different
	// color if you want.
	RGBColor eraseColor;
	eraseColor.red = 0;
	eraseColor.green = 0;
	eraseColor.blue = 0;
			
	// The two zeroes tell QuickTime to maintain the screen resolution.
	// The nil tells QuickTime not to create a window.  We handle
	// the window ourselves.  The zero at the end means we're not
	// passing any flags to the BeginFullScreen function
	error = BeginFullScreen(&quickTimeStateData, GetMainDevice(), 0, 0,
				nil, &eraseColor, 0);
}


// Missile list functions
void GameApp::LaunchMissile(GameSpritePtr attacker, GameWeaponPtr attackingWeapon)
{
	short startX = DetermineMissileStartX(attacker);
	short startY = DetermineMissileStartY(attacker);

	//double startX = DetermineMissileStartX(attacker);
	//double startY = DetermineMissileStartY(attacker);
    
	SpriteDirection attackerDirection = attacker->GetSpriteDirection();

    // We have only one type of weapon, a crossbow bolt,
    // so we launch it.  If you had multiple missile types,
    // you would want to determine the type of missile first,
    // then launch that particular missile.
    LaunchBolt(startX, startY, attackerDirection);
    NotifyEnemiesOfAttack(startX, startY);
}

short GameApp::DetermineMissileStartX(GameSpritePtr attacker)
{
    // If the attacker faces up or down, the missile starts
    // at the horizontal center of the attacker.  If the attacker
    // faces left, the missile starts at the left edge of the attacker.
    // If the attacker faces right, the missile starts at the right
    // edge of the attacker.
	
    /*
    Vector2D attackerPosition;
    attackerPosition = attacker->GetPosition();
    
    double leftEdge;
    double attackerCenterX;
    double rightEdge;
    
    leftEdge = attackerPosition.GetXComponent();
    attackerCenterX = leftEdge + 8;
    rightEdge = leftEdge + attacker->GetSpriteWidth();
    */
    
    short leftEdge = attacker->GetWorldX(); //- attacker->GetSpriteWidth();
	short attackerCenterX = leftEdge + 8; //(attacker->GetSpriteWidth() / 2);
	short rightEdge = leftEdge + attacker->GetSpriteWidth();
	
    SpriteDirection missileDirection = attacker->GetSpriteDirection();

	switch (missileDirection) {
		case kSpriteFacingUp:		
			return attackerCenterX;
			break;
			
		case kSpriteFacingDown:
			return attackerCenterX;
			break;

		case kSpriteFacingLeft:
			return leftEdge;
			break;

		case kSpriteFacingRight:
			return rightEdge;
			break;
			
		default:
			return 0;
			break;
	}
	
	return 0;
}

short GameApp::DetermineMissileStartY(GameSpritePtr attacker)
{
    // If the attacker faces left or right, the missile starts
    // at the vertical center of the attacker.  If the attacker
    // faces up, the missile starts at the top edge of the attacker.
    // If the attacker faces down, the missile starts at the bottom
    // edge of the attacker.

    /*
    Vector2D attackerPosition;
    attackerPosition = attacker->GetPosition();
    
    double topEdge;
    double attackerCenterY;
    double bottomEdge;
    
    topEdge = attackerPosition.GetYComponent();
    attackerCenterY = topEdge + 8;
    bottomEdge = topEdge + attacker->GetSpriteHeight();
    */
	
    short topEdge = attacker->GetWorldY(); //- attacker->GetSpriteHeight();
	short attackerCenterY = topEdge + 8;  //(attacker->GetSpriteHeight() / 2);
	short bottomEdge = topEdge + attacker->GetSpriteHeight();
    
	SpriteDirection missileDirection = attacker->GetSpriteDirection();
	
	switch (missileDirection) {
		case kSpriteFacingUp:		
			return topEdge;
			break;
			
		case kSpriteFacingDown:
			return bottomEdge;
			break;

		case kSpriteFacingLeft:
			return attackerCenterY;
			break;

		case kSpriteFacingRight:
			return attackerCenterY;
			break;
			
		default:
			return 0;
			break;
	}
	
	return 0;
}

//void GameApp::LaunchBolt(double startX, double startY, SpriteDirection missileDirection)
void GameApp::LaunchBolt(short startX, short startY, SpriteDirection missileDirection)
{
	GameWeaponPtr newMissile;
    newMissile = new GameWeapon();
    
    if (newMissile == nil)
        return;
        
    newMissile->SetWeaponClass(kEdgedWeapon); 
    newMissile->SetRange(kBoltRange);
    newMissile->SetDamage(kBoltDamage);
	
	newMissile->SetSpriteDirection(missileDirection);
    newMissile->SetSpriteRow(kBoltWeaponRow);
	newMissile->SetSpriteStorage(context->GetWeaponSpriteStorage());
	newMissile->SetDestination(context);
	newMissile->InitializeSpriteBlitter();
	
    // Set initial position and mass
    Vector2D startPosition;
    startPosition.SetXComponent((double)startX);
    startPosition.SetYComponent((double)startY);
    newMissile->SetPosition(startPosition);
    newMissile->SetWorldX(startX);
    newMissile->SetWorldY(startY);
    newMissile->SetMass(kBoltMass);
    
    // Set launch force and maximum speed
    newMissile->CalculateInitialForce(missileDirection);
    newMissile->SetMaxSpeed(kBoltMaxSpeed);
    
    // Add missile to list
	missileList.AddItem(newMissile);
}

void GameApp::KillMissile(GameWeaponPtr theMissile)
{
	//theMissile->UpdateDirtyRectTable();
    
    missileList.RemoveItem(theMissile);
	delete theMissile;
	theMissile = nil;
}

void GameApp::DeleteMissileList(void)
{
	Iterator listIterator(&missileList);
	GameWeaponPtr 	nextMissile;
	int itemsToRemove = missileList.GetNumberOfItems();
	
	while (itemsToRemove > 0) {
		nextMissile = (GameWeaponPtr)listIterator.Next();
		KillMissile(nextMissile);
		itemsToRemove--;	
	}
}

// Enemy list functions
void GameApp::AddEnemyToList(LevelResidentPtr enemyToAdd, GameWeaponPtr weaponList,
        		GameContextPtr theDrawContext)
{
    
	GameOffscreenBufferPtr monsterToAddStorage;
	short startPositionX = enemyToAdd->GetStartX();
	short startPositionY = enemyToAdd->GetStartY();

	GameEnemyPtr newEnemy = new GameEnemy;
	if (newEnemy != nil) {
		newEnemy->SetWorldX(startPositionX);
		newEnemy->SetWorldY(startPositionY);
		
		Vector2D startingPosition; //= newEnemy->GetPosition();
		startingPosition.SetXComponent((double)startPositionX);
		startingPosition.SetYComponent((double)startPositionY);
		newEnemy->SetPosition(startingPosition);
		
		newEnemy->SetMass(kEnemyMass);
		newEnemy->SetMaxSpeed(kEnemyMaxSpeed);
		
		newEnemy->SetMood(kPatrollingState);
			
		monsterToAddStorage = theDrawContext->GetEnemySpriteStorage();
		newEnemy->SetSpriteStorage(monsterToAddStorage);
		newEnemy->SetSpriteWidth(kEnemySpriteWidth);
		newEnemy->SetSpriteHeight(kEnemySpriteHeight);
		newEnemy->SetDestination(theDrawContext);
		newEnemy->InitializeSpriteBlitter();
				
		newEnemy->DetermineViewRect(currentLevel, currentTileList);
        newEnemy->SetHitPoints(kEnemyHitPoints);
        newEnemy->SetPointValue(kEnemyPointValue);
		newEnemy->SetEnemyWeapon(&(weaponList[kEnemyAttackIndex]));
		enemyList.AddItem(newEnemy);
	}
	
	/*
	switch (enemyToAdd->GetResident()) {
		case kEnemy:
			GameEnemyPtr newEnemy = new GameEnemy;
			if (newEnemy != nil) {
				newEnemy->SetWorldX(enemyToAdd->GetStartX());
				newEnemy->SetWorldY(enemyToAdd->GetStartY());
				newEnemy->DetermineViewRect(currentLevel, currentTileList);
			
				monsterToAddStorage = theDrawContext->GetEnemySpriteStorage();
				newEnemy->SetSpriteStorage(monsterToAddStorage);
				newEnemy->SetSpriteWidth(kEnemySpriteWidth);
				newEnemy->SetSpriteHeight(kEnemySpriteHeight);
				newEnemy->SetDestination(theDrawContext);
				newEnemy->InitializeSpriteBlitter();
				
				newEnemy->SetEnemyWeapon(&(weaponList[kEnemyAttackIndex]));
				enemyList.AddItem(newEnemy);
			}
			break;
	}
	*/
        
}

void GameApp::RemoveEnemyFromList(GameEnemyPtr enemyToRemove)
{
    
}

void GameApp::DeleteEnemyList(void)
{
    Iterator enemyListIterator(&enemyList);
    GameEnemyPtr currentEnemy;

    int itemsToDispose = enemyList.GetNumberOfItems();
    
    while(itemsToDispose > 0) {        
        currentEnemy = (GameEnemyPtr)enemyListIterator.Next();
        enemyList.RemoveItem(currentEnemy);
        delete currentEnemy;
        currentEnemy = nil;
        itemsToDispose--;
    }
}


// Item list related functions
void GameApp::AddItemToList(LevelResidentPtr itemToAdd, GameContextPtr theDrawContext)
{
	
    GameItemPtr newItem = new GameItem;
	GameOffscreenBufferPtr itemToAddStorage;
	
	newItem->SetWorldX(itemToAdd->GetStartX());
	newItem->SetWorldY(itemToAdd->GetStartY());
	
	switch (itemToAdd->GetResident()) {
		/*
		case kCrossbow:
			itemToAddStorage = theDrawContext->GetItemSpriteStorage();
			newItem->SetWhatItIs(kCrossbow);
			newItem->SetSpriteStorage(itemToAddStorage);
			newItem->SetSpriteWidth(kCrossbowSpriteWidth);
			newItem->SetSpriteHeight(kCrossbowSpriteHeight);
			newItem->SetSpriteRow(kCrossbowSpriteRow);
			newItem->SetSpriteColumn(kCrossbowSpriteColumn);
			newItem->SetDestination(theDrawContext);
			newItem->InitializeSpriteBlitter();
			break;
		*/
		case kBolts:
			itemToAddStorage = theDrawContext->GetItemSpriteStorage();
			newItem->SetWhatItIs(kBolts);
			newItem->SetSpriteStorage(itemToAddStorage);
			newItem->SetSpriteWidth(kItemSpriteWidth);
			newItem->SetSpriteHeight(kItemSpriteHeight);
			newItem->SetSpriteRow(kBoltsSpriteRow);
			newItem->SetSpriteColumn(kBoltsSpriteColumn);
			newItem->SetDestination(theDrawContext);
			newItem->InitializeSpriteBlitter();
			break;


		case kCoins:
			itemToAddStorage = theDrawContext->GetItemSpriteStorage();
			newItem->SetWhatItIs(kCoins);
			newItem->SetSpriteStorage(itemToAddStorage);
			newItem->SetSpriteWidth(kItemSpriteWidth);
			newItem->SetSpriteHeight(kItemSpriteHeight);
			newItem->SetSpriteRow(kCoinsSpriteRow);
			newItem->SetSpriteColumn(kCoinsSpriteColumn);
			newItem->SetDestination(theDrawContext);
			newItem->InitializeSpriteBlitter();
			break;

		case kGems:
			itemToAddStorage = theDrawContext->GetItemSpriteStorage();
			newItem->SetWhatItIs(kGems);
			newItem->SetSpriteStorage(itemToAddStorage);
			newItem->SetSpriteWidth(kItemSpriteWidth);
			newItem->SetSpriteHeight(kItemSpriteHeight);
			newItem->SetSpriteRow(kGemsSpriteRow);
			newItem->SetSpriteColumn(kGemsSpriteColumn);
			newItem->SetDestination(theDrawContext);
			newItem->InitializeSpriteBlitter();
			break;

		case kHealingPotion:
			itemToAddStorage = theDrawContext->GetItemSpriteStorage();
			newItem->SetWhatItIs(kHealingPotion);
			newItem->SetSpriteStorage(itemToAddStorage);
			newItem->SetSpriteWidth(kItemSpriteWidth);
			newItem->SetSpriteHeight(kItemSpriteHeight);
			newItem->SetSpriteRow(kHealingPotionSpriteRow);
			newItem->SetSpriteColumn(kHealingPotionSpriteColumn);
			newItem->SetDestination(theDrawContext);
			newItem->InitializeSpriteBlitter();
			break;

		default:
		 	break;
	}
	
	// Add item to item list
	itemList.AddItem(newItem);
    
}

void GameApp::RemoveItemFromList(GameItemPtr itemToRemove)
{
        
}

void GameApp::DeleteItemList(void)
{
    Iterator itemListIterator(&itemList);
    GameItemPtr currentItem;

    int itemsToDispose = itemList.GetNumberOfItems();
    
    while(itemsToDispose > 0) {        
        currentItem = (GameItemPtr)itemListIterator.Next();
        itemList.RemoveItem(currentItem);
        delete currentItem;
        currentItem = nil;
        itemsToDispose--;
    }

}

// Weapon List functions
void GameApp::FillWeaponList(void)
{
	weaponList[kCrossbowIndex].SetWeaponClass(kEdgedWeapon);
	weaponList[kCrossbowIndex].SetRange(kCrossbowRange);
	weaponList[kCrossbowIndex].SetDamage(kCrossbowDamage);

	weaponList[kBoltIndex].SetWeaponClass(kEdgedWeapon);
	weaponList[kBoltIndex].SetRange(kBoltRange);
	weaponList[kBoltIndex].SetDamage(kBoltDamage);

	weaponList[kEnemyAttackIndex].SetWeaponClass(kBluntWeapon);
	weaponList[kEnemyAttackIndex].SetRange(kEnemyAttackRange);
	weaponList[kEnemyAttackIndex].SetDamage(kEnemyAttackDamage);
	
}

// Enemy notification functions
void GameApp::NotifyEnemiesOfIntruders(void)
{
	// When the player moves, we check to see if
	// he moves into one of the enemies' field of view.
	// If so, send it a message that there is an intruder.
	Iterator enemyListIterator(&enemyList);
	GameEnemyPtr currentEnemy = nil;
	//FiniteStateMachinePtr stateMachine;
	
	short playerCenterX = player1->GetWorldX() + (kPlayerSpriteWidth / 2);
	short playerCenterY = player1->GetWorldY() + (kPlayerSpriteHeight / 2);

	do {
		// listIterator.Next provides us the link to the current missile
		currentEnemy = (GameEnemyPtr)enemyListIterator.Next();
		if (currentEnemy != nil) {
			enemyAIController->SetModelToControl(currentEnemy);
			if (enemyAIController->IsInViewRect(playerCenterX, playerCenterY)) {
				//*stateMachine = enemyAIController->GetDecisionMaker();
				//stateMachine->HandleStateTransition(kPlayerEntersView);
				enemyAIController->HandleStateTransition(kPlayerEntersView);
			}
		}
	} while (currentEnemy != nil);
	
}

void GameApp::NotifyEnemiesOfAttack(short missileX, short missileY)
{
	// When the player attacks, check to see which 
	// enemies would be able to notice the attack.
	// Let those enemies know they are under attack.
	Iterator enemyListIterator(&enemyList);
	GameEnemyPtr currentEnemy = nil;
	//FiniteStateMachinePtr stateMachine;
	
	do {
		// listIterator.Next provides us the link to the current missile
		currentEnemy = (GameEnemyPtr)enemyListIterator.Next();
		if (currentEnemy != nil) {
			enemyAIController->SetModelToControl(currentEnemy);
			if (enemyAIController->IsInViewRect(missileX, missileY)) {
				//*stateMachine = enemyAIController->GetDecisionMaker();
				//stateMachine->HandleStateTransition(kPlayerAttacks);
				enemyAIController->HandleStateTransition(kPlayerAttacks);
			}
		}
	} while (currentEnemy != nil);
	
}

// Creation functions
GameWeaponPtr GameApp::CreateWeapon(void)
{
    GameWeaponPtr result;
    
    result = new GameWeapon;
    
    return result;
}

// Utility functions
Vector2D GameApp::CalculatePlayerForce(void)
{
    Vector2D result;

    // Force = (mass * axis value from joystick) * 2
    result.SetXComponent(player1InputController->GetXAxisValue() * 
			player1->GetMass() * 200.0);
    result.SetYComponent(player1InputController->GetYAxisValue() * 
			player1->GetMass() * 200.0);
    
	return result;
}


double GameApp::CalculateTimeStep(void)
{
	UnsignedWide currentTime;
	double currentTimeAsDouble;
	
	// Find current time and convert it to a double.
	// The UnsignedWide data type that Microseconds() uses
	// makes it difficult to perform mathematical operations.
	// Converting to a double makes it easier to calculate the elapsed time.
	Microseconds(&currentTime);
	currentTimeAsDouble = ConvertMicrosecondsToDouble(&currentTime);
	
	// Find elapsed time
	double elapsedTime = currentTimeAsDouble - previousTime;
	
	// Calculate time step in terms of seconds
	double oneMicrosecond = .000001;
	double result = oneMicrosecond * elapsedTime;
	
	// Update previousTime for next timestep calculation.
	previousTime = currentTimeAsDouble;
	return result; 
}

double GameApp::ConvertMicrosecondsToDouble(UnsignedWidePtr microsecondsValue)
{
	double twoPower32 = 4294967296.0;
	double result;
	
	double upperHalf = (double) microsecondsValue->hi;
    double lowerHalf = (double) microsecondsValue->lo;
    result = (upperHalf * twoPower32) + lowerHalf;
	return result;
}

short GameApp::GetCurrentColorDepth(void)
{
    GDHandle	mainDevice = NULL;
    mainDevice = GetMainDevice();
    
    PixMapHandle screenPixelMap = (**mainDevice).gdPMap;
    short currentColorDepth = GetPixDepth(screenPixelMap);

    return currentColorDepth;
}

Boolean GameApp::IsResidentAnEnemy(LevelResidentPtr residentToTest)
{
	switch (residentToTest->GetResident()) {
		case kEnemy:
        // You would add cases for your all game's enemies here
			return true;
			break;
			
		default:
			return false;
			break;
	}
}

// Non-member functions
void SetMenuAbility(MenuHandle menu, short menuItem, Boolean enabling)
{
	if (enabling)
		EnableMenuItem(menu, menuItem);
	else
		DisableMenuItem(menu, menuItem);
		
}

Boolean WasKeyPressed(short keyCode, KeyMap theKeyboard)
{    
	short keymapIndex;
	short bitInIndex;
	short keymapEntrySize = 8;
 
	// The keymap consists of 16 8-bit numbers.
    // Find which of the 16 numbers to check.
    keymapIndex = keyCode / keymapEntrySize;
	
    // Find which bit to check in the keymap index.
    bitInIndex = keyCode % keymapEntrySize;

    // Raise the number 2 by the power of
    // the value of bitInIndex to determine
    // the number we should test in the keymap
    // to see if the key was pressed.  The left
    // shift operator (<<) performs a multiply by 2.
    short keyTestValue;
    keyTestValue = 1 << bitInIndex;
    
    // Get the value of the keymap entry
    // we want to test
    char* startOfKeymap =  (char *) &theKeyboard[0];
    char keymapEntryValue = * (startOfKeymap + keymapIndex);
    
    // Compare the keymap entry with the value
    // of the key we want to test to determine
    // if the player pressed the key.
    return ((keymapEntryValue & keyTestValue) != 0);
    
}
