// FiniteStateMachine.cp	by Mark Szymczyk

#ifndef FINITE_STATE_MACHINE
    #include "FiniteStateMachine.h"
#endif

// Constructor
FiniteStateMachine::FiniteStateMachine(void)
{
    stateTable = nil;
    stateTotal = 0;
    currentStateIndex = 0;
}

// Copy constructor
FiniteStateMachine::FiniteStateMachine(const FiniteStateMachine& theMachine)
{
    // Because the state table is a pointer, we can't
    // just assign the table like the following statement:
    //
    // stateTable = theMachine.stateTable;
    //
    // When one of the FiniteStateMachine objects is deleted,
    // the other object's state table will point to nothing.
    // We must allocate the state table, then call FillStateTable().
    
    if (stateTable != nil) {
        delete [] stateTable;
        stateTable = nil;
    }

	stateTable = new FiniteState[theMachine.stateTotal];
	FillStateTable(theMachine.stateTable);	    	
    stateTotal = theMachine.stateTotal;
    currentStateIndex = theMachine.currentStateIndex;
}

// Destructor
FiniteStateMachine::~FiniteStateMachine(void)
{
    if (stateTable != nil) {
        delete [] stateTable;
        stateTable = nil;
    }
}

// Overload assignment operator
// It is good practice to overload the = operator
// in classes where you dynamically allocate memory.
FiniteStateMachine& FiniteStateMachine::operator= (const FiniteStateMachine& theMachine)
{
    if (this != &theMachine) {		// Avoid self assignment
        stateTotal = theMachine.stateTotal;
    
        // Delete any previosuly allocated state tables.
        if (stateTable != nil)
            delete [] stateTable;
        
        stateTable = new FiniteState[theMachine.stateTotal];
        FillStateTable(theMachine.stateTable);
        //stateTotal = theMachine.stateTotal;
        currentStateIndex = theMachine.currentStateIndex;
    }
    
    return *this;        
}

// Accessors
short FiniteStateMachine::GetStateTotal(void)
{
    return stateTotal;
}

void FiniteStateMachine::SetStateTotal(short theTotal)
{
    stateTotal = theTotal;
}
        
short FiniteStateMachine::GetCurrentStateIndex(void)
{
    return currentStateIndex;
}

void FiniteStateMachine::SetCurrentStateIndex(short theIndex)
{
    currentStateIndex = theIndex;
}


// Filling the machine with states
void FiniteStateMachine::FillStateTable(FiniteStatePtr tableToAdd)
{
	// The function assumes that the state table has already
	// been allocated.
	
	if (stateTable == nil)
		return;
		
	if (tableToAdd == nil)
		return;
	
	for (short index = 0; index < stateTotal; index++)
		AddState(&tableToAdd[index], index);
}

void FiniteStateMachine::AddState(FiniteStatePtr stateToAdd, short index)
{
	// The index parameter tells us where to add the 
	// state in the table

	if (stateToAdd == nil)
		return;

	// Make sure we dont go past the array bounds
	if (index < 0)
		return;
	else if (index >= stateTotal)
		return;

	stateTable[index] = *stateToAdd;
}


// Handling state transitions
void FiniteStateMachine::HandleStateTransition(StateTransitionType theTransition)
{
	StateType theNewState;

    // Have the state handle the transition
    theNewState = stateTable[currentStateIndex].HandleTransition(theTransition);

    // Find the new state in the state table and
    // set current state to the index of the new state
    StateType currentStateType;
    for (short index = 0; index < stateTotal; index++) {
		currentStateType = stateTable[index].GetState();
		if (currentStateType == theNewState) {
			SetCurrentStateIndex(index);
			return;
		}
	}

    // We should never get here, but if we do, well do nothing,
    // which means the state remains unchanged
}


// Utility functions
short FiniteStateMachine::GetIndexFromState(StateType theState)
{
	short result;
		
	switch (theState) {
		case kStandingState:
			result = 0;
            break;

		case kPatrollingState:
			result = 1;
			break;
			
		case kChasingState:
			result = 2;
			break;
			
		case kFleeingState:
			result = 3;
			break;
		
        default:	
			result = 0;		
			break;
	}
	
	
	return result;

}
