// FiniteState.cp	by Mark Szymczyk

#ifndef FINITE_STATE
    #include "FiniteState.h"
#endif

// Constructor
FiniteState::FiniteState(void)
{
    transitionTable = nil;
    transitionTotal = 0;
}

// Copy constructor
FiniteState::FiniteState(const FiniteState& theState)
{
    // Because the transition table is a pointer, we can't
    // just assign the table like the following statement:
    //
    //	transitionTable = theState.transitionTable;
	//
    // When one of the FiniteState objects is deleted,
    // the other object's transition table will point to nothing.
    // We must allocate the transition table, then call FillTransitionTable().

	if (transitionTable != nil) { 
        delete [] transitionTable;
        transitionTable = nil;
    }
    
	transitionTable = new StateTransition[theState.transitionTotal];
	FillTransitionTable(theState.transitionTable);
    transitionTotal = theState.transitionTotal;
    state = theState.state;
}

// Destructor
FiniteState::~FiniteState(void)
{
    if (transitionTable != nil) {
        delete [] transitionTable;
        transitionTable = nil;
    }
}

// Overload assignment operator
// We must do this to avoid an error on exit
// when we dispose of the transition table.
FiniteState& FiniteState::operator= (const FiniteState& theState)
{
    if (this != &theState) {		// Avoid self assignment
        transitionTotal = theState.transitionTotal;

        // delete any previously allocated memory.
        if (transitionTable != nil)
            delete [] transitionTable;
        
        transitionTable = new StateTransition[theState.transitionTotal];
        FillTransitionTable(theState.transitionTable);
        //transitionTotal = theState.transitionTotal;
        state = theState.state;
    }
    
    return *this;        
}

// Accessors
short FiniteState::GetTransitionTotal(void)
{
    return transitionTotal;
}

void FiniteState::SetTransitionTotal(short theTotal)
{
    transitionTotal = theTotal;
}
        
StateType FiniteState::GetState(void)
{
    return state;
}

void FiniteState::SetState(StateType theState)
{
    state = theState;
}

// Filling the transition table
void FiniteState::FillTransitionTable(StateTransitionPtr tableToAdd)
{
	// The function assumes that the transition table has already
	// been allocated.
	
	if (transitionTable == nil)
		return;
		
	if (tableToAdd == nil)
		return;
	
	for (short index = 0; index < transitionTotal; index++)
		AddTransition(&tableToAdd[index], index);
}

void FiniteState::AddTransition(StateTransitionPtr transitionToAdd, short index)
{
	// The index parameter tells us where to add the 
	// transition in the table

	if (transitionToAdd == nil)
		return;

	// Make sure we dont go past the array bounds
	if (index < 0)
		return;
	else if (index >= transitionTotal)
		return;

	transitionTable[index] = *transitionToAdd;
}

// Dealing with transitions
StateType FiniteState::HandleTransition(StateTransitionType theTransition)
{
	StateTransitionType currentTransition;
	StateType result;

    for (short index = 0; index < transitionTotal; index++) {
		currentTransition = transitionTable[index].GetTransitionTrigger();
		if (currentTransition == theTransition) {
			result = transitionTable[index].GetNewState();
			return result;
		}
	}

	// At this point, we found no matching transition
	// Use the current state and don't perform a transition.
	return (GetState());
	//return kStandingState;
}
