// ConfigurationDialog.cp	by Mark Szymczyk

#ifndef CONFIGURATION_DIALOG
	#include "ConfigurationDialog.h"
#endif


ConfigurationDialog::ConfigurationDialog(void)
{
    currentController = nil;
}

ConfigurationDialog::~ConfigurationDialog(void)
{
    
}


// Accessors
InputControllerPtr ConfigurationDialog::GetCurrentController(void)
{
    return currentController;
}
        
void ConfigurationDialog::SetCurrentController(InputControllerPtr theController)
{
    currentController = theController;
}

InputDevicePtr ConfigurationDialog::GetStoredDevice(void)
{
    return storedDevice;
}
        
void ConfigurationDialog::SetStoredDevice(InputDevicePtr theDevice)
{
    storedDevice = theDevice;
}


// Functions from MovableModalDialog that we override
void ConfigurationDialog::Create(short resourceID)
{
	MovableModalDialog::Create(resourceID);
    SetDialogDefaultItem(dialog, ok);
    SetDialogCancelItem(dialog, cancel);
    
    // Store the selected controller and its device list
    // so we can restore it if the user cancels.
    SetStoredDevice(currentController->GetControllerDevice());
    
    // Fill the contents of the dialog box's menus
    FillChooseControllerMenu(currentController);
    FillControlMenus(currentController);
    DrawDialog(dialog);
}

void ConfigurationDialog::HandleKeyDown(EventRecord* event)
{
	char keyPressed;

	// Get the key pressed
	keyPressed = (char)(event->message & charCodeMask);
			
	// Check for Command-period
	if (event->modifiers & cmdKey) {
		if (keyPressed == '.') 
			DoItemHit(cancel);
	}
	else {	
		// Check for return key
		if (keyPressed == kReturnKeyCharCode || keyPressed == kEnterKeyCharCode) 
			DoItemHit(ok);
		// Check for Esc key
		else if (keyPressed == kEscapeKeyCharCode) 
			DoItemHit(cancel);	
	}

}

void ConfigurationDialog::DoItemHit(short theItem)
{
	switch (theItem) {
		case ok:
			HandleOKSelection();
			break;
			
		case cancel:
			HandleCancelSelection();
			break;
            
        case kChooseControllerMenu:
            HandleControllerMenuSelection();
            break;
            
        case kXAxisMenu:
            HandleXAxisMenuSelection();
            break;
            
        case kYAxisMenu:
            HandleYAxisMenuSelection();
            break;
        
        case kAttackMenu:
            HandleAttackMenuSelection();
            break;

        case kChangeWeaponMenu:
            HandleChangeWeaponMenuSelection();
            break;
            
        default:
            break;
			
	}
	
}

void ConfigurationDialog::HandleOKSelection(void)
{    
    UpdateControllerDevice();
    SetControlNeeds();
    
    // Set up the newly selected controller so we can
    // read data from it
    currentController->CloseDevice();
    currentController->Setup();
    
    // Save preferences in a preferences file here.
    // We will handle this in Chapter 17.
    
    Close();
}

void ConfigurationDialog::HandleCancelSelection(void)
{
    // Restore the selected controller.  If the player chose
    // a different controller, then cancelled, the 
    // controllerDevice data member of the InputController class
    // changes so we need to restore the selected device.
    InputDevicePtr theDevice = GetStoredDevice();
    currentController->SetControllerDevice(theDevice);
    
    // We must recreate the element list in case the player
    // chose a different controller.  If the player chose a new
    // input device before cancelling, the old element list got
    // erased so we could fill the control menus with the new
    // device's events.  Rebuilding the element list ensures
    // we have the current device's elements.
    currentController->CreateElementList(theDevice->GetDeviceObject());
    Close();
}

void ConfigurationDialog::HandleControllerMenuSelection(void)
{
    // Fill the other menus with the selected
    // menu's elements
	MenuRef chooseControllerMenu;
    ControlRef chooseControllerMenuControl;
    Rect chooseControllerMenuControlRect;
    DialogItemType itemType;
	int menuItemSelected;
	Str255 selectedDeviceName;

	// Find the name of the input device the player chose.
    GetDialogItem(dialog, kChooseControllerMenu, &itemType, 
            (Handle *)&chooseControllerMenuControl, &chooseControllerMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (chooseControllerMenuControl != nil)) {
            menuItemSelected = GetControlValue(chooseControllerMenuControl);
            chooseControllerMenu = GetControlPopupMenuHandle(chooseControllerMenuControl); 
            GetMenuItemText(chooseControllerMenu, menuItemSelected, selectedDeviceName);
    }
	else {
		return;
	}

	// Find the input device the player chose from
    // the controller list.
    LinkedList deviceList = currentController->GetControllerList();

    InputDevicePtr currentDevice;
    Iterator deviceListIterator(&deviceList);
    Str255 currentDeviceString;
    
    currentDevice = (InputDevicePtr)deviceListIterator.Next();
    short stringsMatchResult;
    
    while (currentDevice != nil) {
        // Get the name of the input device.
        currentDeviceString = currentDevice->productName;

        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(currentDeviceString, selectedDeviceName);        
        if(stringsMatchResult == kStringsMatch) {
            // This is the device we want
            currentController->SetControllerDevice(currentDevice);
            FillControlMenus(currentController);
            return;
        }
            

        // Get next device in list
        currentDevice = (InputDevicePtr)deviceListIterator.Next();
    }
    
}

void ConfigurationDialog::HandleXAxisMenuSelection(void)
{

}

void ConfigurationDialog::HandleYAxisMenuSelection(void)
{

}

void ConfigurationDialog::HandleAttackMenuSelection(void)
{

}

void ConfigurationDialog::HandleChangeWeaponMenuSelection(void)
{
    
}


// Menu filling functions
// We have no idea what HID controllers the user has connected and
// no idea the number of controls (buttons, dpads, sticks) on
// each device.  These functions fill the menu with
// the information the computer reads from the player's machine.
void ConfigurationDialog::FillControlMenus(InputControllerPtr controller)
{
    // Get element list so we can fill the control menus
    InputDevicePtr selectedDevice = controller->GetControllerDevice();    
    controller->CreateElementList(selectedDevice->GetDeviceObject());
    
    FillXAxisMenu(controller);
    FillYAxisMenu(controller);
    FillAttackMenu(controller);
    FillChangeWeaponMenu(controller);
}

void ConfigurationDialog::FillChooseControllerMenu(InputControllerPtr controller)
{
    controller->CreateControllerList();
    
    LinkedList deviceList = controller->GetControllerList();
    
    MenuRef chooseControllerMenu;
    ControlRef chooseControllerMenuControl;
    //Handle chooseControllerMenu;
    Rect chooseControllerMenuControlRect;
    DialogItemType itemType;
    
    // Clear the menu of any items it has
    GetDialogItem(dialog, kChooseControllerMenu, &itemType, 
            (Handle *)&chooseControllerMenuControl, &chooseControllerMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (chooseControllerMenuControl != nil)) {
        chooseControllerMenu = GetControlPopupMenuHandle(chooseControllerMenuControl);
        DeleteMenuItems(chooseControllerMenu, kFirstItem, 
                        CountMenuItems(chooseControllerMenu));
    }

    // Fill the menu with all the controllers connected
    // to the player's machine.
    InputDevicePtr currentDevice;    
    Iterator deviceListIterator(&deviceList);
    OSStatus result;
    ConstStr255Param deviceString;
    
    currentDevice = (InputDevicePtr)deviceListIterator.Next();

    while (currentDevice != nil) {
        deviceString = currentDevice->productName;

        // Add the name to the configuration menu
        GetDialogItem(dialog, kChooseControllerMenu, &itemType, 
                (Handle *)&chooseControllerMenuControl, &chooseControllerMenuControlRect);
        if ((itemType == (ctrlItem + resCtrl)) && (chooseControllerMenuControl != nil)) {
            chooseControllerMenu = GetControlPopupMenuHandle(chooseControllerMenuControl);
            result = AppendMenuItemText(chooseControllerMenu, deviceString);
            //result = InsertMenuItemText(chooseControllerMenu, deviceString, kInsertAtStartOfMenu);
        }

        // Get next device in list
        currentDevice = (InputDevicePtr)deviceListIterator.Next();
    }
    
    SetControlMaximum(chooseControllerMenuControl, deviceList.GetNumberOfItems());
    
    // Set the selected menu item
    InputDevicePtr chosenDevice = controller->GetControllerDevice();
    Str255 chosenDeviceName;
    chosenDeviceName = chosenDevice->productName;
    Str255 menuText;
    short menuItemCount = CountMenuItems(chooseControllerMenu);
    short stringsMatchResult;
    
    // Go through each item in the menu.  If its device
    // name matches the chosen device name, set the menu
    // to that item.
    for (short index = 1; index <= menuItemCount; index++) {
        GetMenuItemText(chooseControllerMenu, index, menuText);
        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(menuText, chosenDeviceName);
        if (stringsMatchResult == kStringsMatch)
            SetControlValue(chooseControllerMenuControl, index);
    }
    
    ShowControl(chooseControllerMenuControl);
}

void ConfigurationDialog::FillXAxisMenu(InputControllerPtr controller)
{
    MenuRef xAxisMenu;
    ControlRef xAxisMenuControl;
    Rect xAxisMenuControlRect;
    DialogItemType itemType;

    // Clear the menu of any items it has
    GetDialogItem(dialog, kXAxisMenu, &itemType, 
            (Handle *)&xAxisMenuControl, &xAxisMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (xAxisMenuControl != nil)) {
        xAxisMenu = GetControlPopupMenuHandle(xAxisMenuControl);
        DeleteMenuItems(xAxisMenu, kFirstItem, CountMenuItems(xAxisMenu));
    }

    // Fill the menu with the elements of the selected input device.
    LinkedList elementList = controller->GetElementList(); 
            
    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    int elementsToFill = elementList.GetNumberOfItems();

    OSStatus result;
    ConstStr255Param currentElementName;
    short menuItemToSelect;
    
    while(elementsToFill > 0) {
        // Get element name
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        currentElementName = currentInputElement->elementName;
        
        // If you wanted to limit the elements that appear in the menu,
        // you would do so here.
        
        // Add the element name to the menu
        GetDialogItem(dialog, kXAxisMenu, &itemType, (Handle *)&xAxisMenuControl, 
                        &xAxisMenuControlRect);
        if ((itemType == (ctrlItem + resCtrl)) && (xAxisMenuControl != nil)) {
            xAxisMenu = GetControlPopupMenuHandle(xAxisMenuControl);
            result = InsertMenuItemText(xAxisMenu, currentElementName, kInsertAtStartOfMenu);
            
            // This code assumes we insert the menu items
            // at the start of the menu.
            if (currentInputElement->GetCookie() == controller->gameNeeds[kXAxisNeed]) {
                menuItemToSelect = elementsToFill;
            }
        }

        elementsToFill--;
    }
    
    SetControlMaximum(xAxisMenuControl, elementList.GetNumberOfItems());
    SetControlValue(xAxisMenuControl, menuItemToSelect);
    ShowControl(xAxisMenuControl);
}
        
void ConfigurationDialog::FillYAxisMenu(InputControllerPtr controller)
{
    MenuRef yAxisMenu;
    ControlRef yAxisMenuControl;
    Rect yAxisMenuControlRect;
    DialogItemType itemType;

    // Clear the menu of any items it has
    GetDialogItem(dialog, kYAxisMenu, &itemType, 
            (Handle *)&yAxisMenuControl, &yAxisMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (yAxisMenuControl != nil)) {
        yAxisMenu = GetControlPopupMenuHandle(yAxisMenuControl);
        DeleteMenuItems(yAxisMenu, kFirstItem, CountMenuItems(yAxisMenu));
    }

    // Fill the menu with the selected device's elements.
    LinkedList elementList = controller->GetElementList(); 
    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    int elementsToFill = elementList.GetNumberOfItems();

    OSStatus result;
    ConstStr255Param currentElementName;
    short menuItemToSelect;
    
    while(elementsToFill > 0) {
        // Get element name
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        currentElementName = currentInputElement->elementName;

        // If you wanted to limit the elements that appear in the menu,
        // you would do so here.
        
        // Add the element name to the menu
        GetDialogItem(dialog, kYAxisMenu, &itemType, (Handle *)&yAxisMenuControl, 
                        &yAxisMenuControlRect);
        if ((itemType == (ctrlItem + resCtrl)) && (yAxisMenu != nil)) {
            yAxisMenu = GetControlPopupMenuHandle(yAxisMenuControl);
            result = InsertMenuItemText(yAxisMenu, currentElementName, kInsertAtStartOfMenu);
            //result = AppendMenuItemText(yAxisMenu, currentElementName);

            // This code assumes we insert the menu items
            // at the start of the menu.
            if (currentInputElement->GetCookie() == controller->gameNeeds[kYAxisNeed]) {
                menuItemToSelect = elementsToFill;
            }
        }

        elementsToFill--;
    }

    SetControlMaximum(yAxisMenuControl, elementList.GetNumberOfItems());
    SetControlValue(yAxisMenuControl, menuItemToSelect);
    ShowControl(yAxisMenuControl);

}
        
void ConfigurationDialog::FillAttackMenu(InputControllerPtr controller)
{
    MenuRef attackMenu;
    ControlRef attackMenuControl;
    Rect attackMenuControlRect;
    DialogItemType itemType;

    // Clear the menu of any items it has
    GetDialogItem(dialog, kAttackMenu, &itemType, 
            (Handle *)&attackMenuControl, &attackMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (attackMenuControl != nil)) {
        attackMenu = GetControlPopupMenuHandle(attackMenuControl);
        DeleteMenuItems(attackMenu, kFirstItem, CountMenuItems(attackMenu));
    }

    // Fill the menu with the selected device's elements.
    LinkedList elementList = controller->GetElementList(); 
            
    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    int elementsToFill = elementList.GetNumberOfItems();

    OSStatus result;
    ConstStr255Param currentElementName;
    short menuItemToSelect;
    
    while(elementsToFill > 0) {
        // Get element name
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        currentElementName = currentInputElement->elementName;

        // If you wanted to limit the elements that appear in the menu,
        // you would do so here.
        
        // Add the element name to the menu
        GetDialogItem(dialog, kAttackMenu, &itemType, (Handle *)&attackMenuControl, 
                        &attackMenuControlRect);
        if ((itemType == (ctrlItem + resCtrl)) && (attackMenu != nil)) {
            attackMenu = GetControlPopupMenuHandle(attackMenuControl);
            result = InsertMenuItemText(attackMenu, currentElementName, kInsertAtStartOfMenu);
            //result = AppendMenuItemText(attackMenu, currentElementName);
            
            // This code assumes we insert the menu items
            // at the start of the menu.
            if (currentInputElement->GetCookie() == controller->gameNeeds[kAttackNeed]) {
                menuItemToSelect = elementsToFill;
            }

        }

        elementsToFill--;
    }

    SetControlMaximum(attackMenuControl, elementList.GetNumberOfItems());
    SetControlValue(attackMenuControl, menuItemToSelect);
    ShowControl(attackMenuControl);

}
        
void ConfigurationDialog::FillChangeWeaponMenu(InputControllerPtr controller)
{
    MenuRef changeWeaponMenu;
    ControlRef changeWeaponMenuControl;
    Rect changeWeaponMenuControlRect;
    DialogItemType itemType;

    // Clear the menu of any items it has
    GetDialogItem(dialog, kChangeWeaponMenu, &itemType, 
            (Handle *)&changeWeaponMenuControl, &changeWeaponMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (changeWeaponMenuControl != nil)) {
        changeWeaponMenu = GetControlPopupMenuHandle(changeWeaponMenuControl);
        DeleteMenuItems(changeWeaponMenu, kFirstItem, CountMenuItems(changeWeaponMenu));
    }

    // Fill the menu with the selected device's elements.
    LinkedList elementList = controller->GetElementList(); 
            //controller->CreateElementList(selectedDevice);
    Iterator elementListIterator(&elementList);
    InputDeviceElementPtr currentInputElement;
    int elementsToFill = elementList.GetNumberOfItems();

    OSStatus result;
    ConstStr255Param currentElementName;
    short menuItemToSelect;
    
    while(elementsToFill > 0) {
        // Get element name
        currentInputElement = (InputDeviceElementPtr)elementListIterator.Next();
        currentElementName = currentInputElement->elementName;

        // If you wanted to limit the elements that appear in the menu,
        // you would do so here.
        
        // Add the element name to the menu
        GetDialogItem(dialog, kChangeWeaponMenu, &itemType, (Handle *)&changeWeaponMenuControl, 
                        &changeWeaponMenuControlRect);
        if ((itemType == (ctrlItem + resCtrl)) && (changeWeaponMenu != nil)) {
            changeWeaponMenu = GetControlPopupMenuHandle(changeWeaponMenuControl);
            result = InsertMenuItemText(changeWeaponMenu, currentElementName, kInsertAtStartOfMenu);
            //result = AppendMenuItemText(changeWeaponMenu, currentElementName);

            // This code assumes we insert the menu items
            // at the start of the menu.
            if (currentInputElement->GetCookie() == controller->gameNeeds[kChangeWeaponNeed]) {
                menuItemToSelect = elementsToFill;
            }
        }
        
        elementsToFill--;
    }

    SetControlMaximum(changeWeaponMenuControl, elementList.GetNumberOfItems());
    SetControlValue(changeWeaponMenuControl, menuItemToSelect);
    ShowControl(changeWeaponMenuControl);

}


// Moving data from the dialog box to the InputController class
void ConfigurationDialog::UpdateControllerDevice(void)
{
    // Set the controller to play the game to the
    // device the player selected in the configuration dialog.

	MenuRef controllerMenu;
    ControlRef controllerMenuControl;
    Rect controllerMenuControlRect;
    DialogItemType itemType;
	int menuItemSelected;
	Str255 selectedDeviceName;

    // Find the name of the device the player selected
    GetDialogItem(dialog, kChooseControllerMenu, &itemType, 
            (Handle *)&controllerMenuControl, &controllerMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (controllerMenuControl != nil)) {
        menuItemSelected = GetControlValue(controllerMenuControl);
        controllerMenu = GetControlPopupMenuHandle(controllerMenuControl);
        GetMenuItemText(controllerMenu, menuItemSelected, selectedDeviceName); 
    }
	else {
		return;
	}

	LinkedList deviceList = currentController->GetControllerList(); 
  
	InputDevicePtr currentDevice;
	Iterator deviceListIterator(&deviceList);
    
	currentDevice = (InputDevicePtr)deviceListIterator.Next();
    short stringsMatchResult;
    // Go through the device list looking for the matching device
	// If we find it set the controller to use to that device.
    while (currentDevice != nil) {
        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(currentDevice->productName, selectedDeviceName);
        if (stringsMatchResult == kStringsMatch) {
            currentController->SetControllerDevice(currentDevice);
            return;
        }
        
        // Retrieve the next device if there is one.
        currentDevice = (InputDevicePtr)deviceListIterator.Next();
    }

}
        
void ConfigurationDialog::SetControlNeeds(void)
{
	SetXAxisControlNeed();
	SetYAxisControlNeed();
	SetAttackControlNeed();
	SetChangeWeaponControlNeed();    
}

void ConfigurationDialog::SetXAxisControlNeed(void)
{
	MenuRef xAxisMenu;
    ControlRef xAxisMenuControl;
    Rect xAxisMenuControlRect;
    DialogItemType itemType;
	int menuItemSelected;
	Str255 selectedElementName;

    // Find the name of the element the player selected
    GetDialogItem(dialog, kXAxisMenu, &itemType, (Handle *)&xAxisMenuControl, &xAxisMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (xAxisMenuControl != nil)) {
        menuItemSelected = GetControlValue(xAxisMenuControl);
        xAxisMenu = GetControlPopupMenuHandle(xAxisMenuControl);
        GetMenuItemText(xAxisMenu, menuItemSelected, selectedElementName); 
    }
	else {
		return;
	}

	LinkedList elementList = currentController->GetElementList(); 

	InputDeviceElementPtr currentElement;
	Iterator elementListIterator(&elementList);
    
	currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    short stringsMatchResult;

    // Go through the element list looking for the matching element
	// If we find it set the controller objects gameNeeds array entry.
    while (currentElement != nil) {
        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(currentElement->elementName, selectedElementName);
        if (stringsMatchResult == kStringsMatch) {
            currentController->gameNeeds[kXAxisNeed] = currentElement->GetCookie();
            return;
        }
        
        // Retrieve the next element if there is one.
        currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    }

}

void ConfigurationDialog::SetYAxisControlNeed(void)
{
	MenuRef yAxisMenu;
    ControlRef yAxisMenuControl;
    Rect yAxisMenuControlRect;
    DialogItemType itemType;
	int menuItemSelected;
	Str255 selectedElementName;

    // Find the name of the element the player selected
    GetDialogItem(dialog, kYAxisMenu, &itemType, (Handle *)&yAxisMenuControl, &yAxisMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (yAxisMenuControl != nil)) {
        menuItemSelected = GetControlValue(yAxisMenuControl);
        yAxisMenu = GetControlPopupMenuHandle(yAxisMenuControl);
        GetMenuItemText(yAxisMenu, menuItemSelected, selectedElementName); 
    }
	else {
		return;
	}

	LinkedList elementList = currentController->GetElementList(); 

	InputDeviceElementPtr currentElement;
	Iterator elementListIterator(&elementList);
    
	currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    short stringsMatchResult;

    // Go through the element list looking for the matching element
	// If we find it set the controller objects gameNeeds array entry.
    while (currentElement != nil) {
        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(currentElement->elementName, selectedElementName);
        if (stringsMatchResult == kStringsMatch) {
            currentController->gameNeeds[kYAxisNeed] = currentElement->GetCookie();
            return;
        }
        
        // Retrieve the next element if there is one.
        currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    }

}

void ConfigurationDialog::SetAttackControlNeed(void)
{
	MenuRef attackMenu;
    ControlRef attackMenuControl;
    Rect attackMenuControlRect;
    DialogItemType itemType;
	int menuItemSelected;
	Str255 selectedElementName;

    // Find the name of the element the player selected
    GetDialogItem(dialog, kAttackMenu, &itemType, (Handle *)&attackMenuControl,
                &attackMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (attackMenuControl != nil)) {
        menuItemSelected = GetControlValue(attackMenuControl);
        attackMenu = GetControlPopupMenuHandle(attackMenuControl);
        GetMenuItemText(attackMenu, menuItemSelected, selectedElementName); 
    }
	else {
		return;
	}

	LinkedList elementList = currentController->GetElementList(); 

	InputDeviceElementPtr currentElement;
	Iterator elementListIterator(&elementList);
    
	currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    short stringsMatchResult;

    // Go through the element list looking for the matching element
	// If we find it set the controller objects gameNeeds array entry.
    while (currentElement != nil) {
        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(currentElement->elementName, selectedElementName);
        if (stringsMatchResult == kStringsMatch) {
            currentController->gameNeeds[kAttackNeed] = currentElement->GetCookie();
            return;
        }
        
        // Retrieve the next element if there is one.
        currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    }
}

void ConfigurationDialog::SetChangeWeaponControlNeed(void)
{
	MenuRef changeWeaponMenu;
    ControlRef changeWeaponMenuControl;
    Rect changeWeaponMenuControlRect;
    DialogItemType itemType;
	int menuItemSelected;
	Str255 selectedElementName;

    // Find the name of the element the player selected
    GetDialogItem(dialog, kChangeWeaponMenu, &itemType, (Handle *)&changeWeaponMenuControl, 
                &changeWeaponMenuControlRect);
    // The value (ctrlItem + resCtrl) is the item type of
    // a popup menu.
    if ((itemType == (ctrlItem + resCtrl)) && (changeWeaponMenuControl != nil)) {
        menuItemSelected = GetControlValue(changeWeaponMenuControl);
        changeWeaponMenu = GetControlPopupMenuHandle(changeWeaponMenuControl);
        GetMenuItemText(changeWeaponMenu, menuItemSelected, selectedElementName); 
    }
	else {
		return;
	}

	LinkedList elementList = currentController->GetElementList();

	InputDeviceElementPtr currentElement;
	Iterator elementListIterator(&elementList);
    
	currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    short stringsMatchResult;

    // Go through the element list looking for the matching element
	// If we find it set the controller objects gameNeeds array entry.
    while (currentElement != nil) {
        // PLStrcmp() is a Toolbox function that compares two Pascal strings
        stringsMatchResult = PLstrcmp(currentElement->elementName, selectedElementName);
        if (stringsMatchResult == kStringsMatch) {
            currentController->gameNeeds[kChangeWeaponNeed] = currentElement->GetCookie();
            return;
        }
        
        // Retrieve the next element if there is one.
        currentElement = (InputDeviceElementPtr)elementListIterator.Next();
    }

}

