// AnimationController.cp	by Mark Szymczyk

#ifndef ANIMATION_CONTROLLER
	#include "AnimationController.h"
#endif

// Constructor
AnimationController::AnimationController(void)
{
	modelToControl = nil;
}

// Destructor
AnimationController::~AnimationController(void)
{

}

// Accessor functions
GameSpritePtr AnimationController::GetModelToControl(void)
{
    return modelToControl; 
}
    
void AnimationController::SetModelToControl(GameSpritePtr theModel)
{ 
    modelToControl = theModel; 
}

// Animation related functions
void AnimationController::UpdateAnimation(InputControllerAction theAction)
{
	switch(theAction) {
		case kMoveUp:
			HandleMoveUp();
			break;
				
		case kMoveDown:
			HandleMoveDown();
				break;

		case kMoveLeft:
			HandleMoveLeft();
			break;
			
		case kMoveRight:
			HandleMoveRight();
			break;
            
		case kMoveUpAndLeft:
			HandleMoveUp();
			break;

		case kMoveUpAndRight:
			HandleMoveRight();
			break;
				
		case kMoveDownAndRight:
			HandleMoveDown();
				break;

		case kMoveDownAndLeft:
			HandleMoveLeft();
			break;
			        
        case kAttack:
            HandleAttack();
            break;
            
		case kNoButtonsPressed:
        case kNoMovement:
            // If we're in the middle of an animation sequence,
            // complete it, otherwise have the creature stand.
            
            if (CurrentlyFighting())
                HandleAttack();
            else if (CurrentlyMoving())
                HandleMovement(modelToControl->GetSpriteDirection());
            else
                HandleStand();
                
			break;
				
		default:
			break;
	}

	if (IsInScreenArea()) {
		modelToControl->UpdateDirtyRectTable();
	}

}

void AnimationController::HandleMovement(SpriteDirection currentDirection)
{
    switch (currentDirection) {
        case kSpriteFacingUp:
            HandleMoveUp();
            break;
            
        case kSpriteFacingDown:
            HandleMoveDown();
            break;
            
        case kSpriteFacingLeft:
            HandleMoveLeft();
            break;
            
        case kSpriteFacingRight:
            HandleMoveRight();
            break;
    }
}
    
void AnimationController::HandleMoveUp(void)
{
	if (modelToControl == nil)
		return;
		
	SpriteDirection creatureDirection = modelToControl->GetSpriteDirection();
	
	if (creatureDirection == kSpriteFacingUp) {
		// Have creature walk up
		MoveToNextFrame();
		modelToControl->SetAction(kSpriteMoving);
	}
	else {
		// Change creature direction
        modelToControl->SetSpriteDirection(kSpriteFacingUp);
		modelToControl->SetFrame(kFirstFrame);
	}

}

void AnimationController::HandleMoveDown(void)
{
	if (modelToControl == nil)
		return;

	SpriteDirection creatureDirection = modelToControl->GetSpriteDirection();
	
	if (creatureDirection == kSpriteFacingDown) {
		// Have creature walk down
		MoveToNextFrame();
		modelToControl->SetAction(kSpriteMoving);

	}
	else {
		// Change creature direction
		modelToControl->SetSpriteDirection(kSpriteFacingDown);
		modelToControl->SetFrame(kFirstFrame);
	}

}

void AnimationController::HandleMoveLeft(void)
{
	if (modelToControl == nil)
		return;

	SpriteDirection creatureDirection = modelToControl->GetSpriteDirection();
	
	if (creatureDirection == kSpriteFacingLeft) {
		// Have creature walk left
		MoveToNextFrame();
		modelToControl->SetAction(kSpriteMoving);
		
	}
	else {
		// Change creature direction
		modelToControl->SetSpriteDirection(kSpriteFacingLeft);
		modelToControl->SetFrame(kFirstFrame);
	}

}

void AnimationController::HandleMoveRight(void)
{
	if (modelToControl == nil)
		return;

	SpriteDirection creatureDirection = modelToControl->GetSpriteDirection();
	
	if (creatureDirection == kSpriteFacingRight) {
		// Have creature walk right
		MoveToNextFrame();
		modelToControl->SetAction(kSpriteMoving);
		
	}
	else {
		// Change creature direction
        modelToControl->SetSpriteDirection(kSpriteFacingRight);
		modelToControl->SetFrame(kFirstFrame);
	}

}

void AnimationController::HandleAttack(void)
{
	if (modelToControl == nil)
		return;

	SpriteAction creatureAction = modelToControl->GetAction();
	
	if (creatureAction == kSpriteFighting) {
		// Move to next frame in attack sequence
		MoveToNextFrame();
	}
	else {
		modelToControl->SetAction(kSpriteFighting);
		modelToControl->SetFrame(kFirstFrame);
	}
    
}

void AnimationController::HandleStand(void)
{
	if (modelToControl == nil)
		return;
    
	modelToControl->SetAction(kSpriteStanding);
	modelToControl->SetFrame(kStandFrame);
}

void AnimationController::ChangeDirection (SpriteDirection newDirection)
{
	if (modelToControl == nil)
		return;

	modelToControl->SetSpriteDirection(newDirection);
	modelToControl->SetAction(kSpriteStanding);
	modelToControl->SetFrame(kStandFrame);
}

void AnimationController::MoveToNextFrame(void)
{
	if (modelToControl == nil)
		return;

	short framesInSequence = modelToControl->GetFramesInSequence();
	
	short currentFrame = modelToControl->GetFrame();
	currentFrame++;
	
	// If we hit the end of the animation sequence, go to first frame
	if (currentFrame > framesInSequence)
		modelToControl->SetFrame(kFirstFrame);
    else
        modelToControl->SetFrame(currentFrame);
		
}



// Utility functions
Boolean AnimationController::CurrentlyFighting(void)
{
    if ((modelToControl->GetAction() == kSpriteFighting) && 
            (modelToControl->GetFrame() < modelToControl->GetFramesInSequence())) {
            
        return true;
    }
    else {
        return false;
    }
}

Boolean AnimationController::CurrentlyMoving(void)
{
    if ((modelToControl->GetAction() == kSpriteMoving) && 
            (modelToControl->GetFrame() < modelToControl->GetFramesInSequence())) {
            
        return true;
    }
    else {
        return false;
    }

}

Boolean AnimationController::CanBeDrawn(GameLevelPtr theLevel)
{
	if (modelToControl == nil)
		return false;

	// We test if the center's tile is visible.
	// If so, we then check if the sprite is in the screen area.
	
	short tileX = (modelToControl->GetWorldX() + 
		(modelToControl->GetSpriteWidth() / 2)) / kTileWidth;
	short tileY = (modelToControl->GetWorldY()) + 
		(modelToControl->GetSpriteHeight() / 2) / kTileHeight;
	
	short theLevelWidth = theLevel->GetLevelWidth();
	UInt32 mapIndex;
	
	mapIndex = (tileY * theLevelWidth) + tileX;
	if (theLevel->levelMap[mapIndex].IsVisibleToPlayer() == false)
		return false;
	else
		return (IsInScreenArea());	
	
}

Boolean AnimationController::IsInScreenArea(void)
{
	// Returns false if entire sprite is not in the viewing area
	// of the screen.  Returns true otherwise.

	if (modelToControl == nil)
		return false;
	
	Rect destRect;
	GameContextPtr theDestination = modelToControl->GetDestination();
	if (theDestination == nil)
		return false;
		
	destRect.top = modelToControl->GetWorldY() - 
		(theDestination->GetVOffset() * kTileHeight);
	destRect.bottom = destRect.top + modelToControl->GetSpriteHeight();
	destRect.left = modelToControl->GetWorldX() - 
		(theDestination->GetHOffset() * kTileWidth);
	destRect.right = destRect.left + modelToControl->GetSpriteWidth();

				
	// Check all 4 edges to see if the sprite is visible onscreen
	if (destRect.bottom <= kBufferTopEdge) {
		return false;		
	}
	
	short bufferHeight = theDestination->GetScreenHeight();
	if (destRect.top >= bufferHeight) {
		return false;		
	}
	
	if (destRect.right <= kBufferLeftEdge) {
		return false;		
	}

	short bufferWidth = theDestination->GetScreenWidth();	
	if (destRect.left >= bufferWidth) {
		return false;		
	}
	
	return true;
}

