// AIController.cp	by Mark Szymczyk

#ifndef AI_CONTROLLER
	#include "AIController.h"
#endif

// Constructor
AIController::AIController(void)
{
	modelToControl = nil;
}

// Destructor
AIController::~AIController(void)
{

}

// Accessor functions
GameEnemyPtr AIController::GetModelToControl(void)
{
    return modelToControl;
}

void AIController::SetModelToControl(GameEnemyPtr theModel)
{
    modelToControl = theModel;
}
        
FiniteStateMachine AIController::GetDecisionMaker(void)
{
    return decisionMaker;
}

void AIController::SetDecisionMaker(FiniteStateMachine theDecisionMaker)
{
    decisionMaker = theDecisionMaker;
}
        
/*
Pathfinder AIController::GetPathMaker(void)
{
    return pathMaker;
}

void AIController::SetPathMaker(Pathfinder thePathMaker)
{
    pathMaker = thePathMaker;
}
*/

void AIController::BuildDecisionMaker(void)
{
	// It would be better to store the states
	// and transitions on disk and read them in, but
	// I'm hard coding it for now.
	
	// Transitions from the patrolling state
	StateTransitionPtr patrollingTransitions = new StateTransition[kNumberOfPatrollingTransitions];
	if (patrollingTransitions == nil)
		return;
	
	patrollingTransitions[0].SetTransitionTrigger(kPlayerEntersView);
	patrollingTransitions[0].SetNewState(kChasingState);
	patrollingTransitions[1].SetTransitionTrigger(kPlayerAttacks);
	patrollingTransitions[1].SetNewState(kChasingState);
	
	// Transitions from the standing state
	StateTransitionPtr standingTransitions = new StateTransition[kNumberOfStandingTransitions];
	if (standingTransitions == nil)
		return;
	
	standingTransitions[0].SetTransitionTrigger(kPlayerEntersView);
	standingTransitions[0].SetNewState(kChasingState);
	standingTransitions[1].SetTransitionTrigger(kPlayerAttacks);
	standingTransitions[1].SetNewState(kChasingState);

	// Transitions from the chasing state
	StateTransitionPtr chasingTransitions = new StateTransition[kNumberOfChasingTransitions];
	if (chasingTransitions == nil)
		return;
	
	chasingTransitions[0].SetTransitionTrigger(kSeverelyWounded); //(kPlayerEntersView);
	chasingTransitions[0].SetNewState(kFleeingState); //(kChasingState);
	chasingTransitions[1].SetTransitionTrigger(kPlayerEscapes); //(kPlayerAttacks);
	chasingTransitions[1].SetNewState(kPatrollingState); //(kChasingState);

	// Transitions from the fleeing state
	StateTransitionPtr fleeingTransitions = new StateTransition[kNumberOfFleeingTransitions];
	if (fleeingTransitions == nil)
		return;
	
	fleeingTransitions[0].SetTransitionTrigger(kEvadedPlayer); //(kPlayerEntersView);
	fleeingTransitions[0].SetNewState(kStandingState); //(kChasingState);
	fleeingTransitions[1].SetTransitionTrigger(kReceivedHelp); //(kPlayerAttacks);
	fleeingTransitions[1].SetNewState(kChasingState);
	
	FiniteStatePtr gameStates = new FiniteState[kStatesInGame];
	if (gameStates == nil)
		return;

	// Fill each state's transition table
	gameStates[kPatrollingState].SetState(kPatrollingState);
	gameStates[kPatrollingState].SetTransitionTotal(kNumberOfPatrollingTransitions);
    gameStates[kPatrollingState].transitionTable = new 
            StateTransition[kNumberOfPatrollingTransitions];
	gameStates[kPatrollingState].FillTransitionTable(patrollingTransitions);

	gameStates[kStandingState].SetState(kStandingState);
	gameStates[kStandingState].SetTransitionTotal(kNumberOfStandingTransitions);
    gameStates[kStandingState].transitionTable = new 
            StateTransition[kNumberOfStandingTransitions];
	gameStates[kStandingState].FillTransitionTable(standingTransitions);

	gameStates[kChasingState].SetState(kChasingState);
	gameStates[kChasingState].SetTransitionTotal(kNumberOfChasingTransitions);
    gameStates[kChasingState].transitionTable = new 
            StateTransition[kNumberOfChasingTransitions];
	gameStates[kChasingState].FillTransitionTable(chasingTransitions);

	gameStates[kFleeingState].SetState(kFleeingState);
	gameStates[kFleeingState].SetTransitionTotal(kNumberOfFleeingTransitions);
    gameStates[kFleeingState].transitionTable = new 
            StateTransition[kNumberOfFleeingTransitions];
	gameStates[kFleeingState].FillTransitionTable(fleeingTransitions);
	
	// Fill the state machine's state table
	decisionMaker.SetStateTotal(kStatesInGame);
    decisionMaker.stateTable = new FiniteState[decisionMaker.GetStateTotal()];
	decisionMaker.FillStateTable(gameStates);
	decisionMaker.SetCurrentStateIndex(kPatrollingState);

	// Dispose of the memory we allocated
	delete [] patrollingTransitions;
	delete [] standingTransitions;
	delete [] chasingTransitions;
	delete [] fleeingTransitions;
	delete [] gameStates;
}

void AIController::HandleStateTransition(StateTransitionType theTransition)
{
	decisionMaker.HandleStateTransition(theTransition);
	
	FiniteStatePtr currentState = &decisionMaker.stateTable[decisionMaker.GetCurrentStateIndex()];
	
	modelToControl->SetMood(currentState->GetState());

}


// Performing AI maneuvers 
InputControllerAction AIController::PerformManeuver(short playerX, short playerY, 
        GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	InputControllerAction result;

	if (modelToControl == nil)
		return kNoMovement;

	//result = Patrol();	

    //decisionMaker.SetCurrentStateIndex(modelToControl->GetMood());
    	
    //short currentStateIndex = decisionMaker.GetCurrentStateIndex();
	//StateType theCurrentState = decisionMaker.stateTable[currentStateIndex].GetState();
    StateType theCurrentState = modelToControl->GetMood();
    
	switch (theCurrentState) {
		case kStandingState:
			result = kNoMovement;
            break;

		case kPatrollingState:
			result = Patrol();
			break;
			
		case kChasingState:
			result = Chase(playerX, playerY, theLevel, theTiles);
			break;
			
		case kFleeingState:
			result = Flee(playerX, playerY, theLevel, theTiles);
			break;
		
        default:	
			result = kNoMovement;		
			break;
	}
	
	return result;
}

Boolean AIController::CanAttack(short preyX, short preyY)
{
	if (modelToControl == nil)
		return false;

	short verticalDistance;
	short horizontalDistance;
	short distanceToPrey;

	short topEdge = modelToControl->GetWorldY();
	short bottomEdge = topEdge + modelToControl->GetSpriteHeight();
	short leftEdge = modelToControl->GetWorldX();
	short rightEdge = leftEdge + modelToControl->GetSpriteWidth();
	
	if (preyY < topEdge) {
		// Prey is above the monster
		verticalDistance = (topEdge - preyY) / kTileHeight;
	}
	else if (preyY > bottomEdge) {
		// Prey is below the monster
		verticalDistance = (preyY - bottomEdge) / kTileHeight;
	}
	else {
		// Prey is within the sprite's boundaries
		verticalDistance = 0;
	}
		
	if (preyX < leftEdge) {
		// Prey is to the left of the monster
		horizontalDistance = (leftEdge - preyX) / kTileWidth;
	}
	else if (preyX > rightEdge) {
		// Prey is to the right of the monster
		horizontalDistance = (preyX - rightEdge) / kTileWidth;
	}
	else {
		// Prey is within the sprite's boundaries
		horizontalDistance = 0;
	}
		
	if (verticalDistance > horizontalDistance)
		distanceToPrey = verticalDistance;
	else
		distanceToPrey = horizontalDistance;
	
	GameWeaponPtr modelWeapon = modelToControl->GetEnemyWeapon();
	if (modelWeapon == nil)
		return false;
		
	if (distanceToPrey <= modelWeapon->GetRange())
		return true;
	else
		return false;
		
}

InputControllerAction AIController::Patrol(void)
{
	// A monster patrols the perimeter of its view rectangle.  It
	// moves in its current direction until it either can't move
	// or moves outside the view rectangle.  When either of these
	// conditions occur, the monster turns left or right.

	if (modelToControl == nil)
		return kNoMovement;

	switch (modelToControl->GetSpriteDirection()) {
		case kSpriteFacingUp:
			return (PatrolUp());
			break;
			
		case kSpriteFacingDown:
			return (PatrolDown());
			break;
			
		case kSpriteFacingLeft:
			return (PatrolLeft());
			break;
			
		case kSpriteFacingRight:
			return (PatrolRight());
			break;
			
		default:
			return kNoMovement;
			break;
		
	}

	return kNoMovement;	
}

InputControllerAction AIController::PatrolUp(void)
{
	if (modelToControl == nil)
		return kNoMovement;

	// This version uses a patrol rectangle
	
	short topEdgeOfSprite = modelToControl->GetWorldY();
	
	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 1 tile inside of the
	// view rectangle plus a buffer zone.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect = modelToControl->GetViewRect();
	
	short bufferZone = kTileHeight + kBufferZone;
	
	short patrolBoundary = patrolRect.top + bufferZone;
	// Check to see if the enemy moved outside of the patrol area.
	if (patrolBoundary < topEdgeOfSprite)
		return kMoveUp;
	else 
		return (TurnLeft());	// kMoveLeft;
        
	/*
    short centerX = modelToControl->GetWorldX() + (modelToControl->GetSpriteWidth() / 2);
	short topEdge = modelToControl->GetWorldY();
	
	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 2 tiles inside of the
	// view rectangle.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolArea variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect;
	Rect theViewRect = modelToControl->GetViewRect();
	patrolRect.top = theViewRect.top + (kTileHeight * 2);
	patrolRect.bottom = theViewRect.bottom - (kTileHeight * 2);
	patrolRect.left = theViewRect.left + (kTileWidth * 2);
	patrolRect.right = theViewRect.right - (kTileWidth * 2);
	
	if (IsInPatrolArea(centerX, topEdge, patrolRect))
		return kMoveUp;
	else
		return (TurnLeft());
    */
}

InputControllerAction AIController::PatrolDown(void)
{
	if (modelToControl == nil)
		return kNoMovement;

	// This version uses a patrol rectangle
	short bottomEdgeOfSprite = modelToControl->GetWorldY() + 
			modelToControl->GetSpriteHeight();

	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 1 tile inside of the
	// view rectangle plus a buffer zone.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect = modelToControl->GetViewRect();
	
	short bufferZone = kTileHeight + kBufferZone;

	short patrolBoundary = patrolRect.bottom - bufferZone;
	// Check to see if the enemy moved outside of the patrol area.
	if (patrolBoundary > bottomEdgeOfSprite)
		return kMoveDown;
	else 
		return (TurnLeft());	// kMoveRight;
	
    /*
    short centerX = modelToControl->GetWorldX() + (modelToControl->GetSpriteWidth() / 2);
	short bottomEdge = modelToControl->GetWorldY() + modelToControl->GetSpriteHeight();

	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 2 tiles inside of the
	// view rectangle.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect;
	Rect theViewRect = modelToControl->GetViewRect();
	patrolRect.top = theViewRect.top + (kTileHeight * 2);
	patrolRect.bottom = theViewRect.bottom - (kTileHeight * 2);
	patrolRect.left = theViewRect.left + (kTileWidth * 2);
	patrolRect.right = theViewRect.right - (kTileWidth * 2);
	
	if (IsInPatrolArea(centerX, bottomEdge, patrolRect))
		return kMoveDown;
	else
		return (TurnLeft());
    */
}

InputControllerAction AIController::PatrolLeft(void)
{
	if (modelToControl == nil)
		return kNoMovement;

	// This version uses a patrol rectangle
	
	short leftEdgeOfSprite = modelToControl->GetWorldX();
	
	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 1 tile inside of the
	// view rectangle plus a buffer zone.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect = modelToControl->GetViewRect();

	short bufferZone = kTileWidth + kBufferZone;

	short patrolBoundary = patrolRect.left + bufferZone;
	// Check to see if the enemy moved outside of the patrol area.
	if (patrolBoundary < leftEdgeOfSprite)
		return kMoveLeft;
	else 
		return (TurnLeft());	//kMoveDown;
        
    /*
	short leftEdge = modelToControl->GetWorldX();
	short centerY = modelToControl->GetWorldY() + (modelToControl->GetSpriteHeight() / 2);

	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 2 tiles inside of the
	// view rectangle.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect;
	Rect theViewRect = modelToControl->GetViewRect();
	patrolRect.top = theViewRect.top + (kTileHeight * 2);
	patrolRect.bottom = theViewRect.bottom - (kTileHeight * 2);
	patrolRect.left = theViewRect.left + (kTileWidth * 2);
	patrolRect.right = theViewRect.right - (kTileWidth * 2);
	
	if (IsInPatrolArea(leftEdge, centerY, patrolRect))
		return kMoveLeft;
	else
		return (TurnLeft());
    */
}

InputControllerAction AIController::PatrolRight(void)
{
	if (modelToControl == nil)
		return kNoMovement;

	// This version uses a patrol rectangle
	
	short rightEdgeOfSprite = modelToControl->GetWorldX() + 
			modelToControl->GetSpriteWidth();
	
	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 1 tile inside of the
	// view rectangle plus a buffer zone.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect = modelToControl->GetViewRect();
	
	short bufferZone = kTileWidth + kBufferZone;

	short patrolBoundary = patrolRect.right - bufferZone;
	// Check to see if the enemy moved outside of the patrol area.
	if (patrolBoundary > rightEdgeOfSprite)
		return kMoveRight;
	else 
		return (TurnLeft());	// kMoveUp;
	
    /*
    short rightEdge = modelToControl->GetWorldX() + modelToControl->GetSpriteWidth();
	short centerY = modelToControl->GetWorldY() + (modelToControl->GetSpriteHeight() / 2);

	// Since the view rectangle includes a border of walls,
	// we create a patrol area that is 2 tiles inside of the
	// view rectangle.  The calculation shouldn't be much of
	// a performance hit, but we can create a patrolRect variable
	// in the GameEnemy class if it is a performance hit.
	Rect patrolRect;
	Rect theViewRect = modelToControl->GetViewRect();
	patrolRect.top = theViewRect.top + (kTileHeight * 2);
	patrolRect.bottom = theViewRect.bottom - (kTileHeight * 2);
	patrolRect.left = theViewRect.left + (kTileWidth * 2);
	patrolRect.right = theViewRect.right - (kTileWidth * 2);
	
	if (IsInPatrolArea(rightEdge, centerY, patrolRect))
		return kMoveRight;
	else
		return (TurnLeft());
    */
}
	
InputControllerAction AIController::Chase(short preyX, short preyY, 
        GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	if (modelToControl == nil)
		return kNoMovement;

	if (CanAttack(preyX, preyY))
		return kAttack;
	
	// For now, we will just have the enemy run to the
    // prey.  This will improve in a later version.
	return MoveModel(preyX, preyY);
}

InputControllerAction AIController::Flee(short predatorX, short predatorY, 
        GameLevelPtr theLevel, GameTileListPtr theTiles)
{
	if (modelToControl == nil)
		return kNoMovement;
	
	// For now, we will just have the enemy run away from the
    // predator.  This will improve in a later version.
    return MoveModelAway(predatorX, predatorY);

}

InputControllerAction AIController::MoveModel(short preyX, short preyY)
{
	// This function assumes that there are no obstacles
	// in the way from the model's current position to
	// where the prey is.

	if (modelToControl == nil)
		return kNoMovement;

	short centerX = modelToControl->GetWorldX() + (modelToControl->GetSpriteWidth() / 2);
	short centerY = modelToControl->GetWorldY() + (modelToControl->GetSpriteHeight() / 2);

	// Compare monster's position to the prey's and act accordingly.
	if ((preyX < centerX) && (preyY < centerY))
		return kMoveUpAndLeft;
	else if ((preyX > centerX) && (preyY < centerY))
		return kMoveUpAndRight;
	else if ((preyX < centerX) && (preyY > centerY))
		return kMoveDownAndLeft;
	else if ((preyX > centerX) && (preyY > centerY))
		return kMoveDownAndRight;
	// At this point, we know that either the preyX or preyY
	// is within our monster sprite's boundaries.
	else if (preyY < centerY)
		return kMoveUp;
	else if (preyY > centerY)
		return kMoveDown;
	else if (preyX < centerX)
		return kMoveLeft;
	else if (preyX > centerX) 
		return kMoveRight;
	else	
		return kNoMovement;
}

InputControllerAction AIController::MoveModelAway(short predatorX, short predatorY)
{
	// This function assumes that there are no obstacles
	// in the way from the model's current position to
	// where the predator is.

	short centerX = modelToControl->GetWorldX() + (modelToControl->GetSpriteWidth() / 2);
	short centerY = modelToControl->GetWorldY() + (modelToControl->GetSpriteHeight() / 2);

	// Compare monster's position to the predator's and act accordingly.
	if ((predatorX < centerX) && (predatorY < centerY))
		return kMoveDownAndRight;
	else if ((predatorX > centerX) && (predatorY < centerY))
		return kMoveDownAndLeft;
	else if ((predatorX < centerX) && (predatorY > centerY))
		return kMoveUpAndRight;
	else if ((predatorX > centerX) && (predatorY > centerY))
		return kMoveUpAndLeft;
	// At this point, we know that either the predatorX or predatorY
	// is within our monster sprite's boundaries.
	else if (predatorY < centerY)
		return kMoveDown;
	else if (predatorY > centerY)
		return kMoveUp;
	else if (predatorX < centerX)
		return kMoveRight;
	else if (predatorX > centerX) 
		return kMoveLeft;
	else	
		return kNoMovement;
}

InputControllerAction AIController::TurnLeft(void)
{
	if (modelToControl == nil)
		return kNoMovement;

	// Set creature's velocity and acceleration to 0
	Vector2D velocityVector = modelToControl->GetVelocity();
	velocityVector.SetXComponent(kStandingSpeed);
	velocityVector.SetYComponent(kStandingSpeed);
	modelToControl->SetVelocity(velocityVector);
	
	Vector2D accelerationVector = modelToControl->GetAcceleration();
	accelerationVector.SetXComponent(kStandingSpeed);
	accelerationVector.SetYComponent(kStandingSpeed);
	modelToControl->SetAcceleration(accelerationVector);
			
	switch (modelToControl->GetSpriteDirection()) {
		case kSpriteFacingUp:
			return kMoveLeft;
			break;
			
		case kSpriteFacingDown:
			return kMoveRight;
			break;
			
		case kSpriteFacingLeft:
			return kMoveDown;
			break;
			
		case kSpriteFacingRight:
			return kMoveUp;
			break;
			
		default:
			return kNoMovement;
			break;
		
	}
	
	return kNoMovement;
}

InputControllerAction AIController::TurnRight(void)
{
	if (modelToControl == nil)
		return kNoMovement;

	// Set creature's velocity and acceleration to 0
	Vector2D velocityVector = modelToControl->GetVelocity();
	velocityVector.SetXComponent(kStandingSpeed);
	velocityVector.SetYComponent(kStandingSpeed);
	modelToControl->SetVelocity(velocityVector);
	
	Vector2D accelerationVector = modelToControl->GetAcceleration();
	accelerationVector.SetXComponent(kStandingSpeed);
	accelerationVector.SetYComponent(kStandingSpeed);
	modelToControl->SetAcceleration(accelerationVector);
    
	switch (modelToControl->GetSpriteDirection()) {
		case kSpriteFacingUp:
			return kMoveRight;
			break;
			
		case kSpriteFacingDown:
			return kMoveLeft;
			break;
			
		case kSpriteFacingLeft:
			return kMoveUp;
			break;
			
		case kSpriteFacingRight:
			return kMoveDown;
			break;
			
		default:
			return kNoMovement;
			break;
		
	}
	
	return kNoMovement;
}

Boolean AIController::IsInViewRect(short x, short y)
{
	if (modelToControl == nil)
		return false;

	Point thePoint;
	
	thePoint.h = x;
	thePoint.v = y;
	
	Rect theViewRect = modelToControl->GetViewRect();
	return (PtInRect(thePoint, &theViewRect));
}

Boolean AIController::IsInPatrolArea(short x, short y, Rect patrolArea)
{
	if (modelToControl == nil)
		return false;

	
	if (y <= patrolArea.top)
		return false;
		
	if (y >= patrolArea.bottom)
		return false;
		
	if (x <= patrolArea.left)
		return false;
		
	if (x >= patrolArea.right)
		return false;
		
	return true;

	/*
    Point thePoint;
	
	thePoint.h = x;
	thePoint.v = y;
	
	return (PtInRect(thePoint, &patrolArea));
    */
}
