/*
 *  GameTexture.cpp
 *  MGPCode(OpenGL)
 *
 *  Created by Mark Szymczyk.
 *  Copyright 2005 Me and Mark Publishing. All rights reserved.
 *
 */

#ifndef GAME_TEXTURE
    #include "GameTexture.h"
#endif

GLuint GameTexture::textureObjectList = 1;

// Constructor
GameTexture::GameTexture(void)
{
    width = 0;
    height = 0;
    imageData = NULL;
    offscreenBuffer = NULL;
    
}

// Destructor
GameTexture::~GameTexture(void)
{
    if (imageData != NULL) {
        DisposePtr(imageData);
        imageData = NULL;
    }
    
    if (offscreenBuffer != NULL) {
        DisposeGWorld(offscreenBuffer);
        offscreenBuffer = NULL;
    }
    
}

// Accessors
int GameTexture::GetWidth(void)
{
    return width;
}

void GameTexture::SetWidth(int theWidth)
{
    width = theWidth;
}

int GameTexture::GetHeight(void)
{
    return height;
}

void GameTexture::SetHeight(int theHeight)
{
    height = theHeight;
}

GLint GameTexture::GetInternalFormat(void)
{
    return internalFormat;
}

void GameTexture::SetInternalFormat(GLint theFormat)
{
    internalFormat = theFormat;
}

GLenum GameTexture::GetFormat(void)
{
    return format;
}

void GameTexture::SetFormat(GLenum theFormat)
{
    format = theFormat;
}

GLenum GameTexture::GetType(void)
{
    return type;
}

void GameTexture::SetType(GLenum theType)
{
    type = theType;
}

Ptr GameTexture::GetImageData(void)
{
    return imageData;
}

void GameTexture::SetImageData(Ptr theData)
{
    imageData = theData;
}

// Texture loading functions
void GameTexture::Load(CFStringRef filename, CFStringRef extension, CFStringRef subdirectory)
{
    // You can pass the entire name of the file as the filename argument or you can
    // separate the entire name and its file extension into the
    // filename and fileExtension arguments. If you pass the entire name as the
    // filename argument, pass NULL as the fileExtension argument.
    
    // You can normally pass NULL as the subdirectory. If you place all your
    // texture files in a special directory in your application, you can pass
    // the directory name as the subdirectory argument to make searching faster.
    
    ReadFromFile(filename, extension, subdirectory);
    Create();
}

void GameTexture::ReadFromFile(CFStringRef filename, CFStringRef extension, CFStringRef subdirectory)
{
    CFBundleRef mainBundle;
    CFURLRef textureFile;
    ComponentInstance fileImporter;
    OSErr error;
    ImageDescriptionHandle imageInfo;
    ComponentResult err;
    
    // Find the texture map's file in the bundle.
    mainBundle = CFBundleGetMainBundle();
    textureFile = CFBundleCopyResourceURL(mainBundle, filename, extension, subdirectory);
    
    // Create the data reference so we can get the file's graphics importer.
    Handle dataRef;
    OSType dataRefType;
    
    dataRef = NewHandle(sizeof(AliasHandle));
    
    // The second parameter to QTNewDataReferenceFromCFURL is flags.
    // It should be set to 0.
    error = QTNewDataReferenceFromCFURL(textureFile, 0, &dataRef, &dataRefType);
    if(error != noErr) {
        DisposeHandle(dataRef);
        CFRelease(textureFile);
        return;
    }
    
    // Get the importer for the file so we can read the information.
    error = GetGraphicsImporterForDataRef(dataRef, dataRefType, &fileImporter);
    
    // Retrieve information about the image
    imageInfo = (ImageDescriptionHandle)NewHandle(sizeof(ImageDescription));
    err = GraphicsImportGetImageDescription(fileImporter, &imageInfo);
    SetWidth((**imageInfo).width);
    SetHeight((**imageInfo).height);
    
    // Get the size of the decompressed data
    unsigned long imageSize;
    imageSize = width * height * (**imageInfo).depth / 8;
    imageData = NewPtr(imageSize);
    
    // Get the boundary rectangle of the image
    Rect imageRect;
    err = GraphicsImportGetNaturalBounds(fileImporter, &imageRect);
    
    // Create an offscreen buffer to hold the image.
    // Apparently QuickTime requires a GWorld to
    // decompress a texture file.
    long bytesPerRow = width * (**imageInfo).depth / 8;
    error = QTNewGWorldFromPtr(&offscreenBuffer, k32ARGBPixelFormat, &imageRect,
                               NULL, NULL, kNativeEndianPixMap, imageData, bytesPerRow);
    
    // Draw the image into the offscreen buffer
    err = GraphicsImportSetGWorld(fileImporter, offscreenBuffer, NULL);
    err = GraphicsImportDraw(fileImporter);
    
    // Cleanup
    error = CloseComponent(fileImporter);
    DisposeHandle((Handle)imageInfo);
    DisposeHandle(dataRef);
}

void GameTexture::Create(void)
{
    // Creates the texture from the file we loaded earlier.

    glPixelStorei(GL_UNPACK_ALIGNMENT, 1);
    glGenTextures(1, &textureObjectList);
    glBindTexture(GL_TEXTURE_2D, textureObjectList);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_S, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_WRAP_T, GL_REPEAT);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MAG_FILTER, GL_NEAREST);
    glTexParameteri(GL_TEXTURE_2D, GL_TEXTURE_MIN_FILTER, GL_NEAREST);
    
    glTexImage2D(GL_TEXTURE_2D, 0, internalFormat, width, height, 0, format, 
                 type, imageData);    
}


// Drawing functions
void GameTexture::Draw(void)
{
    glEnable(GL_TEXTURE_2D);
    glTexEnvf(GL_TEXTURE_ENV, GL_TEXTURE_ENV_MODE, GL_REPLACE);
    glBindTexture(GL_TEXTURE_2D, textureObjectList);
    
    // Draw the texture
    glBegin(GL_QUADS);
    
    // Lower left corner
    glTexCoord2f(0.0, 0.0);
    glVertex3f(-0.5, 0.5, 0.0);
     
    // Lower right corner
    glTexCoord2f(1.0, 0.0);
    glVertex3f(0.5, 0.5, 0.0);
    
    // Upper right corner
    glTexCoord2f(1.0, 1.0);
    glVertex3f(0.5, -0.5, 0.0);
    
    // Upper left corner
    glTexCoord2f(0.0, 1.0);
    glVertex3f(-0.5, -0.5, 0.0);
    
    glEnd();
    glFlush();
    glDisable(GL_TEXTURE_2D);
}    
