/*
 *  GameApp.cpp
 *  
 *
 *  Created by Mark Szymczyk.
 *  Copyright (c) 2005 Me and Mark Publishing. All rights reserved.
 *
 */

#include "GameApp.h"

// Constructor
GameApp::GameApp(void)
{

}

// Destructor
GameApp::~GameApp(void)
{
    CleanUpApp();
}

void GameApp::InitApp(void)
{
    InitToolbox();
    SetUpMenuBar();
    InitializeAGL();
    InstallKeyboardEventHandler();
    InstallTimer();

    textureMap.SetInternalFormat(GL_RGBA8);
    
    // The call to QTNewGWorldFromPtr() requires a
    // 32-bit ARGB pixel format for 32-bit color. There
    // is no GL_ARGB format so we must use GL_BGRA and
    // reverse the values on big-endian systems.
    textureMap.SetFormat(GL_BGRA);
    
    #if __BIG_ENDIAN__
        textureMap.SetType(GL_UNSIGNED_INT_8_8_8_8_REV);
    #else
        textureMap.SetType(GL_UNSIGNED_INT_8_8_8_8);
    #endif
    
    textureMap.Load(CFSTR("Tiles"), CFSTR("png"), NULL);
}

void GameApp::InitToolbox (void)
{
    // Initialize all the needed managers. 
    // In Carbon, you need to initialize only the cursor.
	
    //InitCursor();
		
}

void GameApp::InitializeAGL(void)
{
    GLint attributeList[] = { AGL_RGBA, AGL_ACCELERATED, AGL_DOUBLEBUFFER, 
        AGL_FULLSCREEN, AGL_NONE };
	AGLPixelFormat 	pixelFormat;
	
    // I had several people report problems with aglChoosePixelFormat() failing
    // on Mac OS X 10.4.7. The problem is that the original code passed NULL as the first
    // parameter and 0 as the second parameter. The AGL documentation says passing NULL as
    // the first parameter is the way to go to support multiple monitor systems, but it doesn't
    // work with the AGL_FULLSCREEN attribute. The fix is to call GetMainDevice() and pass the
    // handle GetMainDevice() returns as the first argument to aglChoosePixelFormat(). I commented
    // out the original line of code in case you need to use it.
    
    // Choose pixel format
    GLenum error;
    
    GDHandle mainScreen = GetMainDevice();
    pixelFormat = aglChoosePixelFormat(&mainScreen, 1, attributeList);
    
    //pixelFormat = aglChoosePixelFormat(NULL, 0, attributeList); 
    error = aglGetError();   
    if (pixelFormat == NULL)
        ExitToShell();
        
    drawContext = aglCreateContext (pixelFormat, NULL);
    error = aglGetError();   
    if (drawContext == NULL)
        ExitToShell();
    
    GLboolean success;
    success = aglSetCurrentContext(drawContext);
    error = aglGetError();   
    
    aglDestroyPixelFormat(pixelFormat);
    error = aglGetError();   

    glShadeModel(GL_FLAT);
}

void GameApp::InstallKeyboardEventHandler(void)
{
    EventTypeSpec keyboardHandlerEvents = { kEventClassKeyboard, kEventRawKeyDown };
    
    InstallApplicationEventHandler(NewEventHandlerUPP(KeyboardEventHandler), kKeyboardEventTotal,
                                   &keyboardHandlerEvents, this, NULL);
    
}

void GameApp::InstallTimer(void)
{
    EventLoopRef mainLoop;
    
    mainLoop = GetMainEventLoop();
    
    gameTimer = NewEventLoopTimerUPP(GameLoopTimer);
    
    // Have game loop run 60 times per second
    // We won't have the timer start until we go fullscreen.
    InstallEventLoopTimer(mainLoop, (kEventDurationForever), (kEventDurationSecond / 60),
            gameTimer, this, &timer);
    
}

void GameApp::CleanUpApp(void)
{
    QuitApplicationEventLoop();
    RemoveEventLoopTimer(timer);
    aglDestroyContext(drawContext);
}


// Event related functions

void GameApp::EventLoop(void)
{
    RunApplicationEventLoop();    
}

// menu related functions

void GameApp::SetUpMenuBar()
{
    OSStatus error;
    IBNibRef menuNib;
     
    // Read menu from nib file main.nib
     
    error = CreateNibReference(CFSTR("main"), &menuNib);
    if (!error)
        SetMenuBarFromNib(menuNib, CFSTR("MenuBar"));
                          
    RegisterMenuCommands();
}

void GameApp::RegisterMenuCommands(void)
{
    // Register our Start/Resume menu item    
    SetMenuItemCommandID(GetMenuRef(kFileMenu), kResumeMenuItem, kCommandResumeGame);
    
    EventTypeSpec menuHandlerEvents = 
        { kEventClassCommand, kEventCommandProcess };
    
    InstallApplicationEventHandler(NewEventHandlerUPP(MenuEventHandler), kEventTotal,
            &menuHandlerEvents, this, NULL);
}

pascal OSStatus GameApp::MenuEventHandler(EventHandlerCallRef myHandlerChain,
        EventRef event, void* userData)
{
    HICommand menuItemChosen;
    OSStatus result;

    GameAppPtr currentApp = (GameAppPtr)userData;
    
    GetEventParameter(event, kEventParamDirectObject, typeHICommand, NULL, 
            sizeof(HICommand), NULL, &menuItemChosen);
            
    // The commandID field of the HICommand data structure tells us
    // the menu command of the menu item the player chose.
    switch (menuItemChosen.commandID) {
        case kCommandResumeGame:
            currentApp->Resume();
            result = noErr;
            break;
            
        case kHICommandQuit:
            currentApp->Quit();
            result = eventNotHandledErr;
            break;
            
        default:
            // The standard application event handler
            // handles any events our menu event handler doesn't handle.
            result = eventNotHandledErr;
            break;
    }
    
    return result;
}

void GameApp::Quit(void)
{
    QuitApplicationEventLoop();
	
}


// Game related functions
pascal void GameApp::GameLoopTimer(EventLoopTimerRef theTimer, void* userData)
{
    GameAppPtr currentApp = (GameAppPtr)userData;
    
    currentApp->GameLoop();
}

pascal OSStatus GameApp::KeyboardEventHandler(EventHandlerCallRef myHandlerChain,
                                          EventRef event, void* userData)
{
    UInt32 keyPressed;
    OSStatus result;
    
    GameAppPtr currentApp = (GameAppPtr)userData;
    
    GetEventParameter(event, kEventParamKeyCode, typeUInt32, NULL, 
                      sizeof(UInt32), NULL, &keyPressed);
    
    switch (keyPressed) {
        case kEscapeKey:
            currentApp->Pause();
            result = noErr;
            break;
            
        default:
            // The standard application event handler
            // handles any events our event handler 
            // doesn't handle.
            result = eventNotHandledErr;
            break;
    }
    
    return result;
}

void GameApp::GameLoop(void)
{
    RenderFrame();
}

void GameApp::RenderFrame(void) 
{
    glClear(GL_COLOR_BUFFER_BIT);
    textureMap.Draw();
    aglSwapBuffers(drawContext);
}

void GameApp::Pause(void)
{
    // Turn off fullscreen
    aglSetDrawable(drawContext, NULL);
    
    DrawMenuBar();
    
    // Shut off the game loop timer
    OSStatus error;
    error = SetEventLoopTimerNextFireTime(timer, (kEventDurationForever));

}

void GameApp::Resume(void)
{
    GLenum error;
    
    aglSetCurrentContext(drawContext);
    error = aglGetError();   

    // Go fullscreen 1024 by 768.
    // 60 is the refresh frequency.
    // 0 is the graphics device index.  This 
    // will work for single monitor systems.
    GLboolean success;
    success = aglSetFullScreen(drawContext, 1024, 768, 60, 0);
    
    // Turn on the game loop timer
    OSStatus error2;
    error2 = SetEventLoopTimerNextFireTime(timer, kEventDurationNoWait);

}
